package updater

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/clock"
)

type (
	UpdateFn   func() error
	updateRule struct {
		key      string
		period   time.Duration
		updateFn UpdateFn
	}

	Updater struct {
		logger  log.Structured
		metrics *updaterMetrics

		rulesChan chan *updateRule
		rules     []*updateRule
	}
)

func NewUpdater(code string, logger log.Logger) *Updater {
	m := newUpdaterMetrics(code)
	return &Updater{
		logger:    logger.Structured(),
		metrics:   m,
		rulesChan: make(chan *updateRule, 10),
	}
}

func (u *Updater) AddUpdateRule(key string, period time.Duration, fn UpdateFn) {
	rule := &updateRule{
		key:      key,
		period:   period,
		updateFn: fn,
	}
	u.rules = append(u.rules, rule)
}

func (u *Updater) UpdateAll(ctx context.Context) error {
	u.logger.Info("start updating")
	defer u.logger.Info("finish updating")

	for _, rule := range u.rules {
		select {
		case <-ctx.Done():
			return nil
		default:
			if err := u.update(rule); err != nil {
				return fmt.Errorf("Updater.UpdateAll: %v", err)
			}
		}
	}
	return nil
}

func (u *Updater) RunUpdating(ctx context.Context) {
	u.logger.Info("start background updating")
	for _, rule := range u.rules {
		go u.assigningLoop(ctx, rule)
	}
	go u.updatingLoop(ctx)
}

func (u Updater) assigningLoop(ctx context.Context, rule *updateRule) {
	ticker := clock.NewTicker(rule.period)
	for {
		select {
		case <-ticker.Chan():
			u.rulesChan <- rule
		case <-ctx.Done():
			return
		}
	}
}

func (u *Updater) updatingLoop(ctx context.Context) {
	for {
		select {
		case rule := <-u.rulesChan:
			if err := u.update(rule); err != nil {
				u.logger.Error(
					"update are failed",
					log.String("update_key", rule.key),
					log.Error(err),
				)
				continue
			}
		case <-ctx.Done():
			return
		}
	}
}

func (u *Updater) update(rule *updateRule) (err error) {
	stats := u.handleStartUpdate(rule.key)
	err = rule.updateFn()

	if xerrors.Is(err, AlreadyUpdated) {
		return nil
	}

	if err != nil {
		u.handleFailedUpdate()
		return xerrors.Errorf("Updater.update: %w", err)
	}
	u.handleFinishUpdate(stats)
	return nil
}

type updateStat struct {
	key   string
	start time.Time
}

func (u *Updater) handleStartUpdate(key string) updateStat {
	u.logger.Info(
		"start updating",
		log.String("update_key", key),
	)
	return updateStat{key: key, start: clock.Now()}
}

func (u *Updater) handleFinishUpdate(stats updateStat) {
	duration := clock.Since(stats.start)

	fields := []log.Field{
		log.String("update_key", stats.key),
		log.Duration("update_duration", duration),
	}

	u.logger.Info("finish update", fields...)
	u.metrics.updateSuccessCount.Inc()
	u.metrics.updateDuration.RecordDuration(duration)
}

func (u *Updater) handleFailedUpdate() {
	u.metrics.updateErrorCount.Inc()
}
