package searcher

import (
	"a.yandex-team.ru/travel/trains/search_api/api/price_calendar"
	date2 "a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/pricecalendar"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"sort"
	"strings"
	"sync"
	"time"

	dictspb "a.yandex-team.ru/travel/proto/dicts/trains"
	"github.com/go-chi/chi/v5/middleware"
	"go.uber.org/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/httputil/client"
	commonModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/common/models"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/mordabackend"
	mbModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/mordabackend/models"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/pathfinderproxy"
	pfpModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/pathfinderproxy/models"
	"a.yandex-team.ru/travel/trains/library/go/httputil/clients/trainapi"
	taModels "a.yandex-team.ru/travel/trains/library/go/httputil/clients/trainapi/models"
	"a.yandex-team.ru/travel/trains/search_api/internal/searcher/models"
)

const (
	ufsLink       = "https://www.ufs-online.ru/kupit-zhd-bilety"
	ufsDomain     = "yandex.ufs-online.ru"
	language      = "ru"
	currencyRub   = "RUB"
	providerP2    = "P2"
	companyCppkID = 153
)

type (
	Config struct {
		TrainAPI                  trainapi.Config
		MordaBackend              mordabackend.Config
		PathfinderProxy           pathfinderproxy.Config
		EnableParallelSearch      bool
		ActivePartnersCheckPeriod time.Duration
		NoTransfersThreshold      int
	}
	uaasExperiments struct {
		NoTransfersThreshold int `json:"TRAINS_NO_TRANSFERS_THRESHOLD,string"`
	}
)

var (
	DefaultConfig = Config{
		TrainAPI:                  trainapi.DefaultConfig,
		MordaBackend:              mordabackend.DefaultConfig,
		PathfinderProxy:           pathfinderproxy.DefaultConfig,
		EnableParallelSearch:      true,
		ActivePartnersCheckPeriod: 10 * time.Second,
		NoTransfersThreshold:      0,
	}
	CarTypesWithPlacesDetails = map[dictspb.TCoachType_EType]bool{
		dictspb.TCoachType_COMPARTMENT: true,
		dictspb.TCoachType_PLATZKARTE:  true,
		dictspb.TCoachType_SUITE:       true,
		dictspb.TCoachType_SOFT:        true,
	}
	CarTypesWithSidePlaces = map[dictspb.TCoachType_EType]bool{
		dictspb.TCoachType_PLATZKARTE: true,
	}
)

type Searcher struct {
	logger log.Logger
	cfg    Config
	ctx    context.Context

	trainAPIClient        *trainapi.TrainAPIClient
	mordaBackendClient    *mordabackend.MordaBackendClient
	pathfinderProxyClient *pathfinderproxy.PathfinderProxyClient
	priceCalendarService  pricecalendar.Service

	activePartners *atomic.Value
	boyEnabled     *atomic.Bool
}

func NewSearcher(ctx context.Context, logger log.Logger, priceCalendarService pricecalendar.Service, cfg Config) (*Searcher, error) {
	const funcName = "searcher.NewSearcher"
	trainAPIClient, err := trainapi.NewTrainAPIClient(&cfg.TrainAPI, logger)
	if err != nil {
		return nil, xerrors.Errorf("%s: NewTrainAPIClient fails: %w", funcName, err)
	}
	mordaBackendClient, err := mordabackend.NewMordaBackendClient(&cfg.MordaBackend, logger)
	if err != nil {
		return nil, xerrors.Errorf("%s: NewMordaBackendClient fails: %w", funcName, err)
	}
	pathfinderProxyClient, err := pathfinderproxy.NewPathfinderProxyClient(ctx, &cfg.PathfinderProxy, logger)
	if err != nil {
		return nil, xerrors.Errorf("%s: NewPathfinderProxyClient fails: %w", funcName, err)
	}

	activePartners := atomic.Value{}
	activePartners.Store([]string{taModels.PartnerCodeIM, taModels.PartnerCodeUFS})

	return &Searcher{
		logger:                logger,
		cfg:                   cfg,
		ctx:                   ctx,
		trainAPIClient:        trainAPIClient,
		mordaBackendClient:    mordaBackendClient,
		pathfinderProxyClient: pathfinderProxyClient,
		priceCalendarService:  priceCalendarService,
		activePartners:        &activePartners,
		boyEnabled:            atomic.NewBool(true),
	}, nil
}

func (s *Searcher) Run(ctx context.Context) {
	const funcName = "searcher.Run"

	go func() {
		ticker := time.NewTicker(s.cfg.ActivePartnersCheckPeriod)
	ForLoop:
		for {
			select {
			case <-ticker.C:
				activePartnersResponse, err := s.trainAPIClient.ActivePartners(ctx, &taModels.ActivePartnersRequest{})
				if err != nil {
					s.logger.Errorf("%s: can not get activePartners: %s", funcName, err.Error())
					continue
				}
				s.activePartners.Store(activePartnersResponse.PartnersCodes)
				for _, partnerCode := range activePartnersResponse.PartnersCodes {
					if partnerCode == taModels.PartnerCodeIM {
						if !s.boyEnabled.Swap(true) {
							s.logger.Info("IM appears in activePartners. BoY enabled")
						}
						continue ForLoop
					}
				}
				if s.boyEnabled.Swap(false) {
					s.logger.Info("no IM in activePartners. BoY disabled")
				}
			case <-ctx.Done():
				return
			}
		}
	}()
}

func (s *Searcher) Search(
	ctx context.Context,
	pointFrom string, pointTo string,
	when string, returnWhen string,
	pinForwardSegmentID string, pinBackwardSegmentID string,
	onlyDirect bool, onlyOwnedPrices bool,
	userArgs models.UserArgs,
	testContext models.TestContext,
) (*models.TransferVariantsResponse, error) {

	roundTrip := returnWhen != ""
	if roundTrip {
		onlyDirect = true
	}

	var forwardErr, backwardErr error
	var forwardVariantsResponse, backwardVariantsResponse *models.TransferVariantsResponse

	forwardSearchTask := func() {
		forwardVariantsResponse, forwardErr = s.oneWaySearch(
			ctx, pointFrom, pointTo, when, onlyDirect, onlyOwnedPrices, pinForwardSegmentID, userArgs, testContext,
		)
	}
	if roundTrip {
		backwardSearchTask := func() {
			backwardVariantsResponse, backwardErr = s.oneWaySearch(
				ctx, pointTo, pointFrom, returnWhen, onlyDirect, onlyOwnedPrices, pinBackwardSegmentID, userArgs, testContext,
			)
		}
		if s.cfg.EnableParallelSearch {
			searchWaitGroup := sync.WaitGroup{}
			searchWaitGroup.Add(1)
			go func() {
				backwardSearchTask()
				searchWaitGroup.Done()
			}()
			forwardSearchTask()
			searchWaitGroup.Wait()
		} else {
			forwardSearchTask()
			backwardSearchTask()
		}
	} else {
		forwardSearchTask()
	}

	if forwardErr != nil {
		return nil, forwardErr
	}

	if !roundTrip {
		return forwardVariantsResponse, nil
	}

	if backwardErr != nil {
		return nil, backwardErr
	}

	return &models.TransferVariantsResponse{
		Context:  forwardVariantsResponse.Context,
		Querying: forwardVariantsResponse.Querying || backwardVariantsResponse.Querying,
		TransferVariants: s.makeRoundtripVariants(
			forwardVariantsResponse.TransferVariants,
			backwardVariantsResponse.TransferVariants,
		),
		NearestDays: models.NearestDates{
			Forward:  forwardVariantsResponse.NearestDays.Forward,
			Backward: backwardVariantsResponse.NearestDays.Forward,
		},
		ActivePartners: s.activePartners.Load().([]string),
	}, nil
}

func (s *Searcher) oneWaySearch(
	ctx context.Context,
	pointFrom string, pointTo string,
	when string,
	onlyDirect bool,
	onlyOwnedPrices bool,
	pinSegmentID string,
	userArgs models.UserArgs,
	testContext models.TestContext,
) (*models.TransferVariantsResponse, error) {
	const funcName = "App.oneWaySearch"
	rid := middleware.GetReqID(ctx)
	header := SetUserArgsHeaders(http.Header{}, userArgs)
	mbRequest := mbModels.SearchRequest{
		RID:           rid,
		PointFrom:     pointFrom,
		PointTo:       pointTo,
		When:          when,
		TransportType: commonModels.TransportTypeTrain,
	}
	mbResponse, mbErr := s.mordaBackendClient.Search(ctx, &mbRequest)

	var searchContext mbModels.ResponseContext
	if mbErr == nil {
		searchContext = mbResponse.Result.Context
		if searchContext.IsChanged {
			pointFrom = searchContext.Search.PointFrom.PointKey
			pointTo = searchContext.Search.PointTo.PointKey
		}
		noTransfersThreshold := s.getNoTransfersThreshold(userArgs)
		if noTransfersThreshold > 0 && len(mbResponse.Result.Segments) >= noTransfersThreshold {
			onlyDirect = true
		}
	}

	boyEnabled := s.boyEnabled.Load()
	taRequest := taModels.TrainTariffsRequest{
		RID:             rid,
		PointFrom:       pointFrom,
		PointTo:         pointTo,
		Dates:           []string{when},
		ExpandedDay:     true,
		NationalVersion: language,
		IncludePriceFee: true,
		ForceUfsOrder:   !boyEnabled,
		MockImPath:      testContext.MockImPath,
		MockImAuto:      testContext.MockImAuto,
	}
	pfpRequest := pfpModels.TransferVariantsWithPricesRequest{
		RID:             rid,
		PointFrom:       pointFrom,
		PointTo:         pointTo,
		When:            when,
		TransportTypes:  []string{commonModels.TransportTypeTrain},
		Language:        language,
		IsBot:           userArgs.IsBot,
		IncludePriceFee: true,
	}
	taWaitGroup := sync.WaitGroup{}
	pfpWaitGroup := sync.WaitGroup{}

	var taResponse *taModels.TrainTariffsResponse
	var pfpResponse *pfpModels.TransferVariantsWithPricesResponse
	var taErr, pfpErr error
	if userArgs.IsBot {
		taResponse, taErr = &taModels.TrainTariffsResponse{}, nil
	} else {
		taWaitGroup.Add(1)
		go func() {
			taResponse, taErr = s.trainAPIClient.TrainTariffs(ctx, &taRequest, header)
			taWaitGroup.Done()
		}()
	}
	if !onlyDirect && boyEnabled {
		pfpWaitGroup.Add(1)
		go func() {
			pfpResponse, pfpErr = s.pathfinderProxyClient.TransfersWithPrices(ctx, &pfpRequest, header)
			pfpWaitGroup.Done()
		}()
	}

	taWaitGroup.Wait()

	var segments []commonModels.SegmentWithTariffs
	if mbErr == nil && taErr == nil {
		if userArgs.IsBot {
			segments = mbResponse.Result.Segments
		} else {
			segments = mergeSegments(mbResponse.Result.Segments, taResponse.Segments)
		}
	} else if taErr == nil {
		segments = taResponse.Segments
		s.logger.Errorf("%s: morda-backend client error: %s", funcName, mbErr.Error())
	} else if mbErr == nil {
		segments = mbResponse.Result.Segments
		s.logger.Errorf("%s: train-api client error: %s", funcName, taErr.Error())
	} else {
		return nil, fmt.Errorf("%s: morda-backend and train-api clients errors: %s, %w", funcName, mbErr.Error(), taErr)
	}

	var (
		querying              = false
		nearestDatesDirection models.NearestDatesDirection
	)
	if taResponse != nil {
		if taResponse.Querying {
			querying = true
		} else {
			if nearestDatesDirectionLocal, err := s.getNearestDatesDirection(ctx, pointFrom, pointTo, when, userArgs); err != nil {
				s.logger.Errorf("%s: %s", funcName, err.Error())
			} else {
				nearestDatesDirection = nearestDatesDirectionLocal
			}
		}
	}
	if mbErr != nil && errors.As(mbErr, new(*client.RetryableError)) {
		querying = true
	}

	variants := make([]models.TransferVariant, len(segments))
	for i, segment := range segments {
		variants[i].Forward = []commonModels.SegmentWithTariffs{segment}
	}

	if !onlyDirect && boyEnabled {
		pfpWaitGroup.Wait()
		if pfpErr == nil {
			for _, transferVariant := range pfpResponse.TransferVariants {
				if len(transferVariant.Segments) == 2 {
					variants = append(variants, makeForwardVariant(transferVariant))
				}
			}
			if pfpResponse.Status == pfpModels.QueryStatusQuerying {
				querying = true
			}
		} else {
			var badRspCodeErr *client.BadResponseCodeError
			if errors.As(pfpErr, &badRspCodeErr) && badRspCodeErr.Code == 204 {
				s.logger.Infof("%s: pathfinder-proxy empty result", funcName)
			} else {
				s.logger.Errorf("%s: pathfinder-proxy client error: %s", funcName, pfpErr.Error())
			}
			if errors.As(pfpErr, new(*client.RetryableError)) {
				querying = true
			}
		}
	}

	enrichedVariants := make([]models.TransferVariant, len(variants))
	enrichedVariantID := 0
	for _, variant := range variants {
		if onlyOwnedPrices {
			clearUnownedTariffs(&variant)
		}
		if enrichedVariant, err := s.enrichTransferVariant(variant); err == nil {
			enrichedVariants[enrichedVariantID] = enrichedVariant
			enrichedVariantID++
		}
		clearPlacesCounters(&variant)
	}
	variants = enrichedVariants[:enrichedVariantID]

	if pinSegmentID != "" {
		pinnedVariants := make([]models.TransferVariant, len(variants))
		pinnedVariantID := 0
		for _, variant := range variants {
			for _, segment := range variant.Forward {
				if segment.ID == pinSegmentID {
					pinnedVariants[pinnedVariantID] = variant
					pinnedVariantID++
					break
				}
			}
		}
		if pinnedVariantID < len(variants) {
			variants = pinnedVariants[:pinnedVariantID]
		}
	}

	return &models.TransferVariantsResponse{
		Context:          searchContext,
		Querying:         querying,
		TransferVariants: s.filterTransferVariants(variants),
		NearestDays: models.NearestDates{
			Forward: nearestDatesDirection,
		},
		ActivePartners: s.activePartners.Load().([]string),
	}, nil
}

func SetUserArgsHeaders(header http.Header, userArgs models.UserArgs) http.Header {
	if userArgs.UaasExperiments != "" {
		header.Set("X-Ya-Uaas-Experiments", userArgs.UaasExperiments)
	}
	if userArgs.Icookie != "" {
		header.Set("X-Ya-ICookie", userArgs.Icookie)
	}
	if userArgs.UserDevice != "" {
		header.Set("X-Ya-User-Device", userArgs.UserDevice)
	}
	if userArgs.YandexUID != "" {
		header.Set("X-Ya-YandexUid", userArgs.YandexUID)
	}
	return header
}

func (s *Searcher) getNearestDatesDirection(
	ctx context.Context,
	pointFrom string, pointTo string, when string,
	userArgs models.UserArgs,
) (models.NearestDatesDirection, error) {
	const (
		funcName             = "Searcher.getNearestDates"
		timeDayDuration      = 24 * time.Hour
		maxOtherDatesCount   = 5
		maxEarlierDaysLookup = 1
		maxLaterDaysLookup   = 14
	)
	nearestDatesDirection := models.NearestDatesDirection{}
	whenTime, err := date2.DateFromString(when)
	if err != nil {
		return nearestDatesDirection, fmt.Errorf("%s: '%s' format error: %w", funcName, when, err)
	}
	calendar, err := s.priceCalendarService.PriceCalendarRange(
		ctx, pointFrom, pointTo,
		date2.DateToString(whenTime.Add(-timeDayDuration*maxEarlierDaysLookup)),
		date2.DateToString(whenTime.Add(timeDayDuration*(maxLaterDaysLookup+1))),
		userArgs,
	)
	if err != nil {
		return nearestDatesDirection, fmt.Errorf("%s: can not get calendar data: %w", funcName, err)
	}
	dates := calendar.Forward.GetDates()

	requestedDateNoDirectTrains := false
	for _, datePrice := range dates {
		dateTime, err := date2.DateFromString(datePrice.GetDate())
		if err != nil {
			s.logger.Errorf("%s: DateFromString error: %s", funcName, err.Error())
			continue
		}
		if dateTime.Before(whenTime.Add(-timeDayDuration * maxEarlierDaysLookup)) {
			continue
		}
		if dateTime.Equal(whenTime) {
			if datePrice.EmptyPriceReason == price_calendar.EmptyPriceReason_EMPTY_PRICE_REASON_NO_DIRECT_TRAINS {
				requestedDateNoDirectTrains = true
				continue
			} else {
				break
			}
		}
		if dateTime.After(whenTime.Add(timeDayDuration * maxLaterDaysLookup)) {
			break
		}

		if datePrice.GetEmptyPriceReason() != price_calendar.EmptyPriceReason_EMPTY_PRICE_REASON_NO_DIRECT_TRAINS {
			nearestDatesDirection.Dates = append(nearestDatesDirection.Dates, models.NearestDate{
				Date: datePrice.GetDate(),
			})
			if len(nearestDatesDirection.Dates) == maxOtherDatesCount {
				break
			}
		}
	}
	if requestedDateNoDirectTrains {
		if len(nearestDatesDirection.Dates) == 0 {
			nearestDatesDirection.Reason = models.NearestDatesReasonNoDirectTrains
		} else {
			nearestDatesDirection.Reason = models.NearestDatesReasonNoRequestedDateDirectTrains
		}
	}
	return nearestDatesDirection, nil
}

func mergeSegments(
	mbSegments []commonModels.SegmentWithTariffs, taSegments []commonModels.SegmentWithTariffs,
) []commonModels.SegmentWithTariffs {
	taSegmentsMap := make(map[string]commonModels.SegmentWithTariffs)
	for _, taSegment := range taSegments {
		taSegmentsMap[taSegment.Key] = taSegment
	}

	segments := make([]commonModels.SegmentWithTariffs, 0, len(mbSegments))
	for _, mbSegment := range mbSegments {
		segment := mbSegment
		for _, key := range mbSegment.TariffsKeys {
			if taSegment, ok := taSegmentsMap[key]; ok {
				segment = mergeSegment(mbSegment, taSegment)
				delete(taSegmentsMap, key)
				break
			}
		}
		segments = append(segments, segment)
	}

	for _, taSegment := range taSegmentsMap {
		if taSegment.CanSupplySegments {
			taSegment.IsDynamic = true
			segments = insertSegment(segments, taSegment)
		}
	}

	return segments
}

func insertSegment(
	segments []commonModels.SegmentWithTariffs, segment commonModels.SegmentWithTariffs,
) []commonModels.SegmentWithTariffs {
	i := sort.Search(len(segments), func(i int) bool {
		return segments[i].Departure.After(segment.Departure)
	})
	if i < len(segments) {
		segments = append(segments, commonModels.SegmentWithTariffs{})
		copy(segments[i+1:], segments[i:])
		segments[i] = segment
	} else {
		segments = append(segments, segment)
	}
	return segments
}

func mergeSegment(
	mbSegment commonModels.SegmentWithTariffs, taSegment commonModels.SegmentWithTariffs,
) commonModels.SegmentWithTariffs {

	if !mbSegment.Departure.Equal(taSegment.Departure) {
		return taSegment
	}

	merged := mbSegment
	merged.OriginalNumber = taSegment.OriginalNumber
	merged.Number = taSegment.Number
	merged.HasDynamicPricing = taSegment.HasDynamicPricing
	merged.Tariffs = taSegment.Tariffs
	merged.Provider = taSegment.Provider
	merged.StationFrom = taSegment.StationFrom
	merged.StationTo = taSegment.StationTo
	if taSegment.Company != (commonModels.Company{}) {
		merged.Company = taSegment.Company
	}

	return merged
}

func makeForwardVariant(variant pfpModels.TransferVariant) models.TransferVariant {
	segments := make([]commonModels.SegmentWithTariffs, len(variant.Segments))
	for i, segment := range variant.Segments {
		segment.Title = segment.Thread.Title
		segment.Number = segment.Thread.Number
		segment.OriginalNumber = segment.Thread.Number
		segments[i] = segment
	}
	return models.TransferVariant{Forward: segments}
}

func (s *Searcher) makeRoundtripVariants(
	forwardVariants []models.TransferVariant,
	backwardVariants []models.TransferVariant,
) []models.TransferVariant {
	transferVariants := make([]models.TransferVariant, len(forwardVariants)*len(backwardVariants))
	transferVariantsCount := 0
	for _, forwardVariant := range forwardVariants {
		for _, backwardVariant := range backwardVariants {
			forwardSegments := forwardVariant.Forward
			backwardSegments := backwardVariant.Forward
			if len(forwardSegments) == 0 || len(backwardSegments) == 0 ||
				!forwardSegments[len(forwardSegments)-1].Arrival.Before(backwardSegments[0].Departure) {
				continue
			}
			variant := models.TransferVariant{
				Forward:  forwardSegments,
				Backward: backwardSegments,
			}
			if enrichedVariant, err := s.enrichTransferVariant(variant); err == nil {
				transferVariants[transferVariantsCount] = enrichedVariant
				transferVariantsCount++
			}
		}
	}
	if transferVariantsCount != len(transferVariants) {
		transferVariants = transferVariants[:transferVariantsCount]
	}
	return transferVariants
}

func (s *Searcher) filterTransferVariants(variants []models.TransferVariant) []models.TransferVariant {
	filtered := make([]models.TransferVariant, len(variants))
	filteredID := 0
	timeNow := time.Now()
VariantLoop:
	for _, variant := range variants {
		for _, direction := range []*[]commonModels.SegmentWithTariffs{&variant.Forward, &variant.Backward} {
			for _, segment := range *direction {
				if len(*direction) > 1 &&
					(segment.Company.ID == companyCppkID || segment.Provider == providerP2) {
					continue VariantLoop
				}
				if segment.Departure.Before(timeNow) {
					continue VariantLoop
				}
			}
		}
		filtered[filteredID] = variant
		filteredID++
	}
	if filteredID < len(variants) {
		filtered = filtered[:filteredID]
	}
	return filtered
}

func (s *Searcher) enrichTransferVariant(variant models.TransferVariant) (models.TransferVariant, error) {
	const funcName = "Searcher.enrichTransferVariant"
	enriched := variant

	if len(enriched.Forward) == 0 {
		return enriched, fmt.Errorf("%s: no segments in enriched", funcName)
	}

	ownerTrains := false
	ownerUfs := false
	minVariantPrice := commonModels.Price{}
	minVariantPriceFails := false
	for _, segments := range [...][]commonModels.SegmentWithTariffs{enriched.Forward, enriched.Backward} {
		for i := range segments {
			segment := &segments[i]

			segment.ID = formatSegmentID(segment)

			minSegmentPrice := commonModels.Price{}
			for _, classPrice := range []commonModels.ClassPrice{
				segment.Tariffs.Classes.Compartment,
				segment.Tariffs.Classes.Suite,
				segment.Tariffs.Classes.Sitting,
				segment.Tariffs.Classes.Platzkarte,
				segment.Tariffs.Classes.Soft,
				segment.Tariffs.Classes.Common,
			} {
				if classPrice.TrainOrderURLOwner == commonModels.TrainOrderURLOwnerTrains {
					ownerTrains = true
				} else if classPrice.TrainOrderURLOwner == commonModels.TrainOrderURLOwnerUfs {
					ownerUfs = true
				}

				if classPrice.Price.Value == 0 {
					continue
				}
				if classPrice.Price.Currency != currencyRub {
					s.logger.Errorf("%s: not supported currency: %s", funcName, classPrice.Price.Currency)
					continue
				}
				if minSegmentPrice.Value == 0 || classPrice.Price.Value < minSegmentPrice.Value {
					minSegmentPrice = classPrice.Price
				}
			}

			if minVariantPriceFails || minSegmentPrice.Value == 0 {
				minVariantPriceFails = true
			} else {
				if minVariantPrice.Value == 0 {
					minVariantPrice = minSegmentPrice
				} else {
					minVariantPrice.Value += minSegmentPrice.Value
				}
			}
		}
	}

	enriched.ID = formatVariantID(&enriched)

	if !minVariantPriceFails {
		enriched.MinPrice = minVariantPrice
	}

	if ownerUfs && !ownerTrains {
		segment := enriched.Forward[0]

		departureTime, err := getLocalDepartureTime(segment)
		if err != nil {
			return enriched, fmt.Errorf("%s: can not evaluate local time: %w", funcName, err)
		}

		values := url.Values{}
		values.Set("domain", ufsDomain)
		values.Set("date", departureTime.Format("2006-01-02"))
		values.Set("trainNumber", segment.Number)

		enriched.OrderURL = models.TransferVariantOrderURL{
			Owner: models.OrderOwnerUFS,
			URL: fmt.Sprintf(
				"%s/%s/%s?%s",
				ufsLink,
				segment.StationFrom.Codes.Express,
				segment.StationTo.Codes.Express,
				values.Encode()),
		}
		return enriched, nil
	}

	if !ownerUfs && ownerTrains {
		enriched.OrderURL = models.TransferVariantOrderURL{
			Owner: models.OrderOwnerTrains,
		}
		return enriched, nil
	}

	if !ownerUfs && !ownerTrains {
		return enriched, nil
	}

	return enriched, fmt.Errorf("%s: more than one class owner", funcName)
}

func (s *Searcher) getNoTransfersThreshold(userArgs models.UserArgs) int {
	const funcName = "Searcher.getNoTransfersThreshold"

	if userArgs.UaasExperiments == "" {
		return s.cfg.NoTransfersThreshold
	}

	experiments := uaasExperiments{}
	if err := json.Unmarshal([]byte(userArgs.UaasExperiments), &experiments); err != nil {
		s.logger.Infof("%s: can not unmarshal UaasExperiments: %s", funcName, err.Error())
	} else {
		if experiments.NoTransfersThreshold != 0 {
			return experiments.NoTransfersThreshold
		}
	}
	return s.cfg.NoTransfersThreshold
}

func clearUnownedTariffs(variant *models.TransferVariant) {
	for _, segments := range [...][]commonModels.SegmentWithTariffs{variant.Forward, variant.Backward} {
		for i := range segments {
			classes := &segments[i].Tariffs.Classes
			for _, classPrice := range []*commonModels.ClassPrice{
				&classes.Compartment,
				&classes.Suite,
				&classes.Sitting,
				&classes.Platzkarte,
				&classes.Soft,
				&classes.Common,
			} {
				if classPrice.TrainOrderURLOwner != commonModels.TrainOrderURLOwnerTrains {
					*classPrice = commonModels.ClassPrice{}
				}
			}
		}
	}
}

func clearPlacesCounters(variant *models.TransferVariant) {
	for _, segments := range [...][]commonModels.SegmentWithTariffs{variant.Forward, variant.Backward} {
		for i := range segments {
			classes := &segments[i].Tariffs.Classes
			for carType, classPrice := range map[dictspb.TCoachType_EType]*commonModels.ClassPrice{
				dictspb.TCoachType_COMPARTMENT: &classes.Compartment,
				dictspb.TCoachType_SUITE:       &classes.Suite,
				dictspb.TCoachType_SITTING:     &classes.Sitting,
				dictspb.TCoachType_PLATZKARTE:  &classes.Platzkarte,
				dictspb.TCoachType_SOFT:        &classes.Soft,
				dictspb.TCoachType_COMMON:      &classes.Common,
			} {
				if _, ok := CarTypesWithPlacesDetails[carType]; !ok {
					classPrice.LowerSeats = nil
					classPrice.UpperSeats = nil
					classPrice.LowerSideSeats = nil
					classPrice.UpperSideSeats = nil
					continue
				}
				if _, ok := CarTypesWithSidePlaces[carType]; !ok {
					classPrice.LowerSideSeats = nil
					classPrice.UpperSideSeats = nil
				}
			}
		}
	}
}

func formatSegmentID(segment *commonModels.SegmentWithTariffs) string {
	return fmt.Sprintf("%d_%d_%d", segment.StationFrom.ID, segment.StationTo.ID, segment.Departure.Unix())
}

func formatVariantID(variant *models.TransferVariant) string {
	idBuilder := strings.Builder{}
	for i, segments := range [...][]commonModels.SegmentWithTariffs{variant.Forward, variant.Backward} {
		for j, segment := range segments {
			if j > 0 {
				idBuilder.WriteByte('_')
			} else if i > 0 {
				idBuilder.WriteByte('-')
			}
			idBuilder.WriteString(segment.ID)
		}
	}
	return idBuilder.String()
}

func getLocalDepartureTime(segment commonModels.SegmentWithTariffs) (time.Time, error) {
	tz, err := time.LoadLocation(segment.StationFrom.Timezone)
	if err != nil {
		return time.Time{}, err
	}
	return segment.Departure.In(tz), nil
}
