package seo

import (
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	tanker "a.yandex-team.ru/travel/library/go/tanker/next"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"github.com/pmezard/go-difflib/difflib"
	"google.golang.org/protobuf/encoding/protojson"

	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo/models"
)

const dataKeyPrefix = "data"

type TemplateChecker struct {
	logger                log.Logger
	directionTestsKeySet  *tanker.KeySet
	rc                    *ResponseConstructor
	onlySupportedLanguage lang.Lang
	checks                []TemplateCheck
	dataMap               map[string]*models.SeoDirectionData
}

type TemplateCheck struct {
	DataIndex    string
	TemplateName string
	Key          string
	Expected     string
}

func NewTemplateChecker(directionTests *tanker.KeySet, logger log.Logger, rc *ResponseConstructor) (*TemplateChecker, error) {
	funcName := "NewTemplateChecker"
	language := lang.Ru
	var checks []TemplateCheck
	dataMap := make(map[string]*models.SeoDirectionData)
	for _, k := range directionTests.GetKeys() {
		ss := strings.Split(k, "-")
		if len(ss) != 2 {
			return nil, fmt.Errorf("%s: invalid \"%s\" key format. must be \"template.name-data.index\" or \"data-data.index\"", funcName, k)
		}
		if ss[0] == dataKeyPrefix {
			s, err := directionTests.GetSingular(k, language.String())
			if err != nil {
				return nil, fmt.Errorf("%s: %w", funcName, err)
			}
			data := &models.SeoDirectionData{}
			err = json.Unmarshal([]byte(s), data)
			if err != nil {
				return nil, fmt.Errorf("%s: invalud %s data format: %w", funcName, k, err)
			}
			prepareData(data)
			dataMap[ss[1]] = data
		} else {
			expected, err := directionTests.GetSingular(k, language.String())
			if err != nil {
				return nil, fmt.Errorf("%s: %w", funcName, err)
			}
			checks = append(checks, TemplateCheck{
				DataIndex:    ss[1],
				TemplateName: ss[0],
				Key:          k,
				Expected:     expected,
			})
		}
	}
	return &TemplateChecker{
		logger:                logger,
		rc:                    rc,
		directionTestsKeySet:  directionTests,
		onlySupportedLanguage: language,
		checks:                checks,
		dataMap:               dataMap,
	}, nil
}

func prepareData(data *models.SeoDirectionData) {
	data.FromPoint = points.NewSettlement(data.FromCity)
	data.ToPoint = points.NewSettlement(data.ToCity)
	if data.FromStation != nil {
		data.FromPoint = points.NewStation(data.FromStation)
	}
	if data.ToStation != nil {
		data.ToPoint = points.NewStation(data.ToStation)
	}
}

func (t *TemplateChecker) RunChecks() error {
	funcName := "TemplateChecker.RunChecks"
	var fails []error
	for _, check := range t.checks {
		err := t.runCheck(check)
		if err == nil {
			t.logger.Infof("%s: %s: ok!", funcName, check.Key)
		} else {
			fails = append(fails, err)
			t.logger.Errorf("%s: %s: failed: %s", funcName, check.Key, err.Error())
		}
	}
	if len(fails) > 0 {
		return fmt.Errorf("%s: failed %v checks of %v", funcName, len(fails), len(t.checks))
	} else {
		t.logger.Infof("%s: %v checks success!", funcName, len(t.checks))
	}
	return nil
}

func (t *TemplateChecker) runCheck(check TemplateCheck) error {
	data, ok := t.dataMap[check.DataIndex]
	if !ok {
		return fmt.Errorf("data with index \"%s\" not found", check.DataIndex)
	}
	if check.TemplateName == KeyScheme {
		rsp, err := t.rc.BuildSeoDirection(data, t.onlySupportedLanguage)
		if err != nil {
			return err
		}
		expected := &seopb.SeoDirectionResponse{}
		err = protojson.Unmarshal([]byte(check.Expected), expected)
		if err != nil {
			return fmt.Errorf("error parse expected response: %w", err)
		}

		rspStr := protojson.Format(rsp)
		expectedStr := protojson.Format(expected)
		err = t.checkStringEquals(rspStr, expectedStr)
		if err != nil {
			return err
		}
	} else {
		res, err := t.rc.tryExecuteTemplate(check.TemplateName, t.onlySupportedLanguage, data)
		if err != nil {
			return err
		}
		err = t.checkStringEquals(res, check.Expected)
		if err != nil {
			return err
		}
	}
	return nil
}

func (t *TemplateChecker) checkStringEquals(actual, expected string) error {
	diff, err := difflib.GetUnifiedDiffString(difflib.UnifiedDiff{
		A:        difflib.SplitLines(actual),
		B:        difflib.SplitLines(expected),
		FromFile: "Actual",
		ToFile:   "Expected",
		Context:  3,
	})
	if err != nil {
		return err
	}
	if diff != "" {
		return fmt.Errorf("diff:\n%s", diff)
	}
	return nil
}
