package seo

import (
	"context"
	"errors"
	"fmt"
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/geotools"
	tanker "a.yandex-team.ru/travel/library/go/tanker/next"

	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/direction/segments"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/date"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/express"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/helpers"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/railway"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/tariffs/cache"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo/models"
)

const (
	timeFormat         = "15:04"
	trainsKey          = "trains"
	aboutTrainsKey     = "trains.about"
	highSpeedTrainsKey = "high.speed.trains"
	brandedTrainsKey   = "branded.trains"
)

type DataCollector struct {
	durationTranslator  *i18n.DurationTranslator
	translatableFactory *i18n.TranslatableFactory
	carTypeTranslator   *i18n.CarTypeTranslator
	commonKeySet        *tanker.KeySet
	tariffCache         *cache.TariffCache
	repoRegistry        *registry.RepositoryRegistry
	expressRepository   *express.Repository
	segmentsMapper      *segments.Mapper
	scheduleRepository  *schedule.Repository
	pointParser         *points.Parser
	logger              log.Logger
}

func NewDataCollector(
	durationTranslator *i18n.DurationTranslator,
	translatableFactory *i18n.TranslatableFactory,
	carTypeTranslator *i18n.CarTypeTranslator,
	commonKeySet *tanker.KeySet,
	tariffCache *cache.TariffCache,
	repoRegistry *registry.RepositoryRegistry,
	expressRepository *express.Repository,
	scheduleRepository *schedule.Repository,
	pointParser *points.Parser,
	logger log.Logger,
) *DataCollector {
	return &DataCollector{
		durationTranslator:  durationTranslator,
		translatableFactory: translatableFactory,
		carTypeTranslator:   carTypeTranslator,
		commonKeySet:        commonKeySet,
		tariffCache:         tariffCache,
		repoRegistry:        repoRegistry,
		expressRepository:   expressRepository,
		segmentsMapper:      segments.NewMapper(logger, repoRegistry),
		scheduleRepository:  scheduleRepository,
		pointParser:         pointParser,
		logger:              logger,
	}
}

func (c *DataCollector) UpdateKeySet(keySet *tanker.KeySet) {
	c.commonKeySet = keySet
}

func (c *DataCollector) BuildDirectionsQuery(fromSlug, toSlug string, language lang.Lang) (DirectionsQuery, error) {
	funcName := "DataCollector.BuildDirectionsQuery"
	from, err := c.pointParser.ParseBySlug(fromSlug)
	if err != nil {
		c.logger.Errorf("%s: not found fromSlug %s: %s", funcName, fromSlug, err.Error())
		return DirectionsQuery{}, &PointNotFoundError{paramName: "fromSlug", slug: fromSlug}
	}
	to, err := c.pointParser.ParseBySlug(toSlug)
	if err != nil {
		c.logger.Errorf("%s: not found toSlug %s: %s", funcName, toSlug, err.Error())
		return DirectionsQuery{}, &PointNotFoundError{paramName: "toSlug", slug: toSlug}
	}
	protoZone, ok := c.repoRegistry.GetTimeZoneRepo().Get(from.TimeZoneID())
	if !ok {
		return DirectionsQuery{}, fmt.Errorf("%s: not found time zone for %v", funcName, fromSlug)
	}
	loc, err := time.LoadLocation(protoZone.Code)
	if err != nil {
		return DirectionsQuery{}, fmt.Errorf("%s: cat not load time zone %s, %s: %w", funcName, protoZone.Code, from.Slug(), err)
	}
	now := time.Now().In(loc)
	leftBorder := time.Date(now.Year(), now.Month(), now.Day()+1, 0, 0, 0, 0, loc)
	return DirectionsQuery{
		From:     from,
		To:       to,
		Language: language,
		FromDate: leftBorder,
	}, nil
}

func (c *DataCollector) GetSeoDirectionsData(ctx context.Context, fromSlug, toSlug string, language lang.Lang) (*models.SeoDirectionData, error) {
	funcName := "DataCollector.GetSeoDirectionsData"
	var query DirectionsQuery
	query, err := c.BuildDirectionsQuery(fromSlug, toSlug, language)
	data := &models.SeoDirectionData{}
	data.Status = seopb.SeoDirectionResponseStatus_RESPONSE_STATUS_OK
	if err != nil {
		return nil, fmt.Errorf("%s: query build failed: %w", funcName, err)
	}
	searchDate, trainSegments, err := c.getSegments(ctx, query)
	if err != nil {
		var emptyDirectionError *EmptyDirectionError
		if errors.As(err, &emptyDirectionError) {
			data.Status = seopb.SeoDirectionResponseStatus_RESPONSE_STATUS_EMPTY_DIRECTION
		} else {
			return nil, fmt.Errorf("%s: segments getting error: %w", funcName, err)
		}
	}
	if searchDate == nil {
		searchDate = &query.FromDate
	}
	data.SearchDate = searchDate.Format(date.DateISOFormat)
	data.ActualYear = searchDate.Year()
	data.FromPoint = query.From
	data.ToPoint = query.To
	stats := c.getSegmentsStats(trainSegments, query.Language)
	data.TrainsCount = stats.trainsCount
	data.FromStationsTitles, err = c.getStationTitles(stats.fromStationsIds, query.Language)
	if err != nil {
		return nil, fmt.Errorf("%s: error getting from station titles %w", funcName, err)
	}
	data.ToStationsTitles, err = c.getStationTitles(stats.toStationsIds, query.Language)
	if err != nil {
		return nil, fmt.Errorf("%s: error getting to station titles %w", funcName, err)
	}
	data.FirstTrainDeparture = stats.firstDeparture.Format(timeFormat)
	data.LastTrainDeparture = stats.lastDeparture.Format(timeFormat)
	for brandTitle := range stats.firmTrains {
		data.FirmTrains = append(data.FirmTrains, brandTitle)
	}
	for brandTitle := range stats.expressTrains {
		data.ExpressTrains = append(data.ExpressTrains, brandTitle)
	}
	data.PricesByCarType = c.updateCarPrices(query, stats.pricesByCarType)
	data.TripRangeKm = getTripRangeKm(query)
	_, data.HasCompartment = stats.pricesByCarType["compartment"]
	_, data.HasPlatzkarte = stats.pricesByCarType["platzkarte"]
	data.MinTripTime, err = c.durationTranslator.FormatHumanDuration(query.Language, stats.minTripTime)
	if err != nil {
		return nil, fmt.Errorf("%s: error format MinTripTime duration: %w", funcName, err)
	}
	data.MaxTripTime, err = c.durationTranslator.FormatHumanDuration(query.Language, stats.maxTripTime)
	if err != nil {
		return nil, fmt.Errorf("%s: error format MaxTripTime duration: %w", funcName, err)
	}
	err = c.updateStrings(query, data)
	if err != nil {
		return nil, fmt.Errorf("%s: error updating strings: %w", funcName, err)
	}
	return data, nil
}

func (c *DataCollector) updateStrings(query DirectionsQuery, data *models.SeoDirectionData) (err error) {
	funcName := "DataCollector.updateStrings"

	fromTitle, err := c.getPointTitle(query.From, query.Language, lang.Nominative)
	if err != nil {
		return fmt.Errorf("%s: error getting from title %w", funcName, err)
	}
	if len(data.FromStationsTitles) == 1 && data.FromStationsTitles[0] == fromTitle {
		data.FromStationsTitles = nil
	}
	toTitle, err := c.getPointTitle(query.To, query.Language, lang.Nominative)
	if err != nil {
		return fmt.Errorf("%s: error getting to title %w", funcName, err)
	}
	if len(data.ToStationsTitles) == 1 && data.ToStationsTitles[0] == toTitle {
		data.ToStationsTitles = nil
	}
	data.TrainsStr, err = c.commonKeySet.GetPlural(trainsKey, query.Language.String(), data.TrainsCount)
	if err != nil {
		return fmt.Errorf("%s: common key-set error, key=%s: %w", funcName, trainsKey, err)
	}
	data.AboutTrainsStr, err = c.commonKeySet.GetPlural(aboutTrainsKey, query.Language.String(), data.TrainsCount)
	if err != nil {
		return fmt.Errorf("%s: common key-set error, key=%s: %w", funcName, aboutTrainsKey, err)
	}
	data.FirmTrainsStr, err = c.commonKeySet.GetSingular(brandedTrainsKey, query.Language.String())
	if err != nil {
		return fmt.Errorf("%s: common key-set error, key=%s: %w", funcName, brandedTrainsKey, err)
	}
	data.ExpressTrainsStr, err = c.commonKeySet.GetSingular(highSpeedTrainsKey, query.Language.String())
	if err != nil {
		return fmt.Errorf("%s: common key-set error, key=%s: %w", funcName, highSpeedTrainsKey, err)
	}
	return nil
}

type trainSegmentsStats struct {
	fromStationsIds map[int32]bool
	toStationsIds   map[int32]bool
	firmTrains      map[string]bool
	expressTrains   map[string]bool
	firstDeparture  time.Time
	lastDeparture   time.Time
	pricesByCarType map[string]*models.CarTypePrices
	minTripTime     time.Duration
	maxTripTime     time.Duration
	trainsCount     int
}

func (c *DataCollector) getSegments(ctx context.Context, query DirectionsQuery) (*time.Time, segments.TrainSegments, error) {
	funcName := "DataCollector.getSegments"
	scheduleSegments := c.scheduleRepository.FindSegments(ctx, query.From, query.To, query.FromDate)
	searchDate, scheduleSegments := schedule.FindFirstDateSegments(scheduleSegments)
	if len(scheduleSegments) == 0 {
		return nil, nil, &EmptyDirectionError{fromSlug: query.From.Slug(), toSlug: query.To.Slug()}
	}
	fromExpressID := int32(c.expressRepository.FindExpressID(query.From))
	toExpressID := int32(c.expressRepository.FindExpressID(query.To))
	tariffs, err := c.tariffCache.Select(ctx,
		[]int32{fromExpressID},
		[]int32{toExpressID},
		searchDate.Add(-time.Hour*24),
		searchDate.Add(time.Hour*24),
	)
	if err != nil {
		return nil, nil, fmt.Errorf("%s: can not load tariffs: %w", funcName, err)
	}
	railWayLocation := railway.GetLocationByPoint(query.From, c.repoRegistry)
	trainSegments := c.segmentsMapper.MapTrainSegments(ctx, railWayLocation, scheduleSegments, tariffs)
	// TODO: apply yandex fee
	return &searchDate, trainSegments, nil
}

func (c *DataCollector) getSegmentsStats(trainSegments segments.TrainSegments, language lang.Lang) trainSegmentsStats {
	funcName := "DataCollector.getSegmentsStats"
	stats := trainSegmentsStats{
		fromStationsIds: make(map[int32]bool),
		toStationsIds:   make(map[int32]bool),
		firmTrains:      make(map[string]bool),
		expressTrains:   make(map[string]bool),
		pricesByCarType: make(map[string]*models.CarTypePrices),
	}
	if len(trainSegments) == 0 {
		return stats
	}
	stats.firstDeparture = trainSegments[0].DepartureLocalDt
	stats.lastDeparture = trainSegments[0].DepartureLocalDt
	stats.minTripTime = trainSegments[0].Duration
	stats.maxTripTime = trainSegments[0].Duration
	for _, s := range trainSegments {
		stats.trainsCount++
		stats.fromStationsIds[s.DepartureStation.StationId] = true
		stats.toStationsIds[s.ArrivalStation.StationId] = true
		if s.TrainBrand != nil {
			brand := c.translatableFactory.TranslatableNamedTrain(s.TrainBrand)
			brandTitle, err := brand.Title(language, lang.Nominative)
			if err != nil {
				c.logger.Warnf("%s: can not get brand title for ID=%v", funcName, s.TrainBrand.Id)
				continue
			}
			if s.TrainBrand.IsHighSpeed {
				stats.expressTrains[brandTitle] = true
			} else if s.TrainBrand.IsDeluxe {
				stats.firmTrains[brandTitle] = true
			}
		}
		for _, p := range s.Places {
			carPrices, ok := stats.pricesByCarType[p.CoachType]
			if ok {
				carPrices.MinPrice = helpers.GetMinPrice(carPrices.MinPrice, p.Price)
			} else {
				stats.pricesByCarType[p.CoachType] = &models.CarTypePrices{
					RawCarType: p.CoachType,
					MinPrice:   p.Price,
				}
			}
		}
		if stats.firstDeparture.After(s.DepartureLocalDt) {
			stats.firstDeparture = s.DepartureLocalDt
		} else if stats.lastDeparture.Before(s.DepartureLocalDt) {
			stats.lastDeparture = s.DepartureLocalDt
		}
		if stats.minTripTime > s.Duration {
			stats.minTripTime = s.Duration
		} else if stats.maxTripTime < s.Duration {
			stats.maxTripTime = s.Duration
		}
	}
	return stats
}

func (c *DataCollector) getStationTitles(stationIds map[int32]bool, language lang.Lang) ([]string, error) {
	var res []string
	for stationID := range stationIds {
		title, err := c.getStationTitle(stationID, language, lang.Nominative)
		if err != nil {
			return nil, err
		}
		res = append(res, title)
	}
	return res, nil
}

func (c *DataCollector) getStationTitle(stationID int32, language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	protoStation, ok := c.repoRegistry.GetStationRepo().Get(stationID)
	if !ok {
		return "", fmt.Errorf("not found station by id %v", stationID)
	}
	station := points.NewStation(protoStation)
	return c.getPointTitle(station, language, grammaticalCase)
}

func (c *DataCollector) getPointTitle(point points.Point, language lang.Lang, grammaticalCase lang.GrammaticalCase) (string, error) {
	title, err := point.Translatable(c.translatableFactory).PopularTitle(language, grammaticalCase)
	if err != nil {
		return "", fmt.Errorf("error getting title: %w", err)
	}
	return title, nil
}

func (c *DataCollector) updateCarPrices(query DirectionsQuery, pricesByCarType map[string]*models.CarTypePrices) []*models.CarTypePrices {
	funcName := "DataCollector.updateCarPrices"
	result := make([]*models.CarTypePrices, 0, 5)
	for carType, carPrice := range pricesByCarType {
		toAccusative, err := c.carTypeTranslator.GetTitleToAccusative(i18n.CarType(carType), query.Language)
		if err != nil {
			c.logger.Warnf("%s: can not get title for car type %s", funcName, carType)
			continue
		}
		carPrice.CarTypeAccusative = toAccusative
		inLocative, err := c.carTypeTranslator.GetTitleInLocative(i18n.CarType(carType), query.Language)
		if err != nil {
			c.logger.Warnf("%s: can not get title for car type %s", funcName, carType)
			continue
		}
		carPrice.CarTypeLocative = inLocative
		carPrice.MinPricePretty = helpers.PrettyIntegerPrice(carPrice.MinPrice)
		result = append(result, carPrice)
	}
	sort.Slice(result, func(i, j int) bool {
		return helpers.GetFloatValue(result[i].MinPrice) < helpers.GetFloatValue(result[j].MinPrice)
	})

	return result
}

func getTripRangeKm(query DirectionsQuery) int {
	return int(geotools.GeoDistance(
		query.From.Latitude(), query.From.Longitude(),
		query.To.Latitude(), query.To.Longitude(),
	) / 1000)
}
