package seo

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	tanker "a.yandex-team.ru/travel/library/go/tanker/next"
	tankerpb "a.yandex-team.ru/travel/proto/tanker"
	"github.com/golang/protobuf/proto"
)

type ResourceUpdatedCallback func() error

type KeySetsResource struct {
	snapshotDirectory string
	logger            log.Logger
	seoResourceReader resourcestorage.StorageReader
	snapshotTS        time.Time
	snapshot          *tankerpb.KeySetExportResponse
	keySets           map[string]*tanker.KeySet
	updateCallback    ResourceUpdatedCallback
}

func NewResource(snapshotDirectory string, logger log.Logger, seoResourceReader resourcestorage.StorageReader) *KeySetsResource {
	return &KeySetsResource{
		snapshotDirectory: snapshotDirectory,
		logger:            logger,
		seoResourceReader: seoResourceReader,
		updateCallback:    nil,
	}
}

func NewResourceFromSnapshot(snapshot *tankerpb.KeySetExportResponse) (*KeySetsResource, error) {
	funcName := "NewResourceFromSnapshot"
	r := &KeySetsResource{
		snapshot: snapshot,
	}
	if err := r.initKeySets(); err != nil {
		return nil, fmt.Errorf("%s: error init sey-sets: %w", funcName, err)
	}
	return r, nil
}

func (r *KeySetsResource) Iter(ctx context.Context) <-chan proto.Message {
	ch := make(chan proto.Message)
	go func() {
		defer close(ch)
		select {
		case ch <- r.snapshot:
			return
		case <-ctx.Done():
			return
		}
	}()
	return ch
}

func (r *KeySetsResource) Add(message proto.Message) {
	r.snapshot = message.(*tankerpb.KeySetExportResponse)
}

func (r *KeySetsResource) initKeySets() error {
	r.keySets = make(map[string]*tanker.KeySet)
	for key, keySetProto := range r.snapshot.Keysets {
		keySet, err := tanker.NewKeySet(keySetProto)
		if err != nil {
			return err
		}
		r.keySets[key] = keySet
	}
	return nil
}

func (r *KeySetsResource) UpdateResource() error {
	funcName := "KeySetsResource.UpdateResource"
	snapshotTime, err := r.seoResourceReader.GetTimestamp(r.snapshotDirectory)
	if err != nil {
		return fmt.Errorf("%s: error getting snapshotTime: %w", funcName, err)
	}
	if snapshotTime.After(r.snapshotTS) {
		resourceLoader := resourcestorage.NewLoader(&tankerpb.KeySetExportResponse{},
			r.snapshotDirectory, r.seoResourceReader, r.logger)
		_, err = resourceLoader.Load(r)
		if err != nil {
			return fmt.Errorf("%s: error load resource: %w", funcName, err)
		}
		r.snapshotTS = snapshotTime
		if err = r.initKeySets(); err != nil {
			return fmt.Errorf("%s: error init sey-sets: %w", funcName, err)
		}
		if r.updateCallback != nil {
			err := r.updateCallback()
			if err != nil {
				return fmt.Errorf("%s: update callback error : %w", funcName, err)
			}
		}
	}
	return nil
}

func (r *KeySetsResource) SetResourceUpdatedCallback(fn ResourceUpdatedCallback) {
	r.updateCallback = fn
}

func (r *KeySetsResource) GetKeySetProto(ksName string) *tankerpb.KeySet {
	return r.snapshot.Keysets[ksName]
}

func (r *KeySetsResource) GetKeySet(ksName string) *tanker.KeySet {
	return r.keySets[ksName]
}
