package seo

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	tanker "a.yandex-team.ru/travel/library/go/tanker/next"
	"a.yandex-team.ru/travel/proto/seo_pages"
	"github.com/flosch/pongo2/v4"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"

	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/templater"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo/models"
)

const (
	KeySetNameCommon         = "trains.common"
	KeySetNameDirection      = "trains.direction-landing"
	KeySetNameDirectionTests = "trains.direction-landing.tests"

	KeyScheme               = "response.scheme"
	KeySchemeEmptyDirection = "response.scheme.empty.direction"

	FallBackLang = lang.Ru
)

var (
	supportedLanguages = []lang.Lang{lang.Ru}
	statusToSchemeKey  = map[seopb.SeoDirectionResponseStatus]string{
		seopb.SeoDirectionResponseStatus_RESPONSE_STATUS_OK:              KeyScheme,
		seopb.SeoDirectionResponseStatus_RESPONSE_STATUS_EMPTY_DIRECTION: KeySchemeEmptyDirection,
	}
	schemeKeys = make(map[string]bool) // values from statusToSchemeKey map
)

func init() {
	for _, schemeKey := range statusToSchemeKey {
		schemeKeys[schemeKey] = true
	}
}

type ResponseConstructor struct {
	logger          log.Logger
	commonKeySet    *tanker.KeySet
	directionKeySet *tanker.KeySet
	schemes         map[lang.Lang]map[seopb.SeoDirectionResponseStatus]*seopb.SeoDirectionResponse
	templates       map[lang.Lang]map[string]*pongo2.Template
}

func NewResponseConstructor(ksCommon, ksDirection *tanker.KeySet, logger log.Logger) (*ResponseConstructor, error) {
	funcName := "NewResponseConstructor"
	schemes := make(map[lang.Lang]map[seopb.SeoDirectionResponseStatus]*seopb.SeoDirectionResponse)
	templates := make(map[lang.Lang]map[string]*pongo2.Template)
	for _, language := range supportedLanguages {
		schemes[language] = make(map[seopb.SeoDirectionResponseStatus]*seopb.SeoDirectionResponse)
		for status, schemeKey := range statusToSchemeKey {
			schemeStr, err := ksDirection.GetSingular(schemeKey, language.String())
			if err != nil {
				return nil, fmt.Errorf("%s: failed get key %s: %w", funcName, schemeKey, err)
			}

			scheme := &seopb.SeoDirectionResponse{}
			err = protojson.Unmarshal([]byte(schemeStr), scheme)
			if err != nil {
				return nil, fmt.Errorf("%s: invalid %s: %w", funcName, schemeKey, err)
			}
			schemes[language][status] = scheme
		}

		templatesMap := make(map[string]*pongo2.Template)
		for _, key := range ksDirection.GetKeys() {
			if _, isSchemeKey := schemeKeys[key]; isSchemeKey {
				continue
			}
			templateStr, err := ksDirection.GetSingular(key, language.String())
			if err != nil {
				return nil, fmt.Errorf("%s: failed get key %s: %w", funcName, key, err)
			}
			template, err := templater.InitTemplate(templateStr)
			if err != nil {
				return nil, fmt.Errorf("%s: InitTemplate failed %s: %w", funcName, key, err)
			}
			templatesMap[key] = template
		}
		templates[language] = templatesMap
	}
	return &ResponseConstructor{
		commonKeySet:    ksCommon,
		directionKeySet: ksDirection,
		schemes:         schemes,
		templates:       templates,
		logger:          logger,
	}, nil
}

func (r *ResponseConstructor) BuildSeoDirection(data *models.SeoDirectionData, language lang.Lang) (*seopb.SeoDirectionResponse, error) {
	funcName := "ResponseConstructor.BuildSeoDirection"
	scheme := r.getResponseScheme(data.Status, language)
	rsp := proto.Clone(scheme).(*seopb.SeoDirectionResponse)
	rsp.SearchDate = data.SearchDate
	rsp.Status = data.Status
	err := r.executeTemplatesOnTrainDirectionResponse(rsp, language, data)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}
	if len(data.PricesByCarType) > 0 && rsp.TrainSeoInfo.OfferData != nil {
		rsp.TrainSeoInfo.OfferData.Price = data.PricesByCarType[0].MinPrice
	} else {
		rsp.TrainSeoInfo.OfferData = nil
	}

	return rsp, nil
}

func (r *ResponseConstructor) getResponseScheme(status seopb.SeoDirectionResponseStatus, language lang.Lang) *seopb.SeoDirectionResponse {
	funcName := "ResponseConstructor.getResponseScheme"
	schemes, ok := r.schemes[language]
	if !ok {
		r.logger.Errorf("%s: language %s is not supported, fallback to %s", funcName, language.String(), FallBackLang.String())
		language = FallBackLang
		schemes = r.schemes[FallBackLang]
	}
	scheme := schemes[status]
	return scheme
}

func (r *ResponseConstructor) executeTemplatesOnTrainDirectionResponse(rsp *seopb.SeoDirectionResponse,
	language lang.Lang, data *models.SeoDirectionData) error {
	funcName := "ResponseConstructor.executeTemplatesOnTrainDirectionResponse"
	if rsp.Header != nil {
		t, err := r.tryExecuteTemplate(rsp.Header.Title, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
		rsp.Header.Title = t

		err = r.tryExecuteTemplatesOnTextBlock(rsp.Header.Text, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
	}

	if rsp.Segments != nil {
		t, err := r.tryExecuteTemplate(rsp.Segments.Title, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
		rsp.Segments.Title = t
	}

	if rsp.Info != nil {
		t, err := r.tryExecuteTemplate(rsp.Info.Title, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
		rsp.Info.Title = t

		err = r.tryExecuteTemplatesOnTextBlock(rsp.Info.Text, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}

		err = r.tryExecuteTemplatesOnTextBlock(rsp.Info.Disclaimer, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
	}

	if rsp.Faq != nil {
		items := make([]*seopb.TrainDirectionFAQItem, 0, len(rsp.Faq.Items))
		for _, faq := range rsp.Faq.Items {
			err := r.tryExecuteTemplatesOnTrainFaq(faq, language, data)
			if err != nil {
				return fmt.Errorf("%s: %w", funcName, err)
			}
			if !r.isEmptyTextBlock(faq.Text) {
				items = append(items, faq)
			}
		}
		rsp.Faq.Items = items
	}

	if rsp.SeoInfo != nil {
		t, err := r.tryExecuteTemplate(rsp.SeoInfo.Title, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
		rsp.SeoInfo.Title = t

		t, err = r.tryExecuteTemplate(rsp.SeoInfo.Description, language, data)
		if err != nil {
			return fmt.Errorf("%s: %w", funcName, err)
		}
		rsp.SeoInfo.Description = t

		if rsp.SeoInfo.OpenGraph != nil {
			t, err := r.tryExecuteTemplate(rsp.SeoInfo.OpenGraph.Title, language, data)
			if err != nil {
				return fmt.Errorf("%s: %w", funcName, err)
			}
			rsp.SeoInfo.OpenGraph.Title = t

			t, err = r.tryExecuteTemplate(rsp.SeoInfo.OpenGraph.Description, language, data)
			if err != nil {
				return fmt.Errorf("%s: %w", funcName, err)
			}
			rsp.SeoInfo.OpenGraph.Description = t
		}
	}

	if rsp.TrainSeoInfo != nil {
		items := make([]*seo_pages.FAQSchemaMarkupItem, 0, len(rsp.TrainSeoInfo.FaqItems))
		for _, faq := range rsp.TrainSeoInfo.FaqItems {
			err := r.tryExecuteTemplatesOnBaseFaq(faq, language, data)
			if err != nil {
				return fmt.Errorf("%s: %w", funcName, err)
			}
			if !r.isEmptyString(faq.Answer) {
				items = append(items, faq)
			}
		}
		rsp.TrainSeoInfo.FaqItems = items
		if rsp.TrainSeoInfo.OfferData != nil {
			t, err := r.tryExecuteTemplate(rsp.TrainSeoInfo.OfferData.Name, language, data)
			if err != nil {
				return fmt.Errorf("%s: %w", funcName, err)
			}
			rsp.TrainSeoInfo.OfferData.Name = t
		}
	}
	return nil
}

func (r *ResponseConstructor) tryExecuteTemplate(templateNameOrString string, language lang.Lang, data interface{}) (string, error) {
	t, ok := r.templates[language][templateNameOrString]
	if ok {
		text, err := templater.ExecuteTemplate(t, data)
		if err != nil {
			return "", fmt.Errorf("error execute template %s: %w", templateNameOrString, err)
		}
		return text, nil
	} else {
		return templateNameOrString, nil
	}
}

func (r *ResponseConstructor) tryExecuteTemplatesOnTextBlock(block *seo_pages.TextBlock, language lang.Lang, data interface{}) error {
	for _, child := range block.Children {
		if child.GetPlainTextBlock() != nil {
			t, err := r.tryExecuteTemplate(child.GetPlainTextBlock().Text, language, data)
			if err != nil {
				return err
			}
			child.GetPlainTextBlock().Text = t
		}
	}
	return nil
}

func (r *ResponseConstructor) isEmptyTextBlock(block *seo_pages.TextBlock) bool {
	empty := true
	for _, child := range block.Children {
		if child.GetPlainTextBlock() != nil {
			if !r.isEmptyString(child.GetPlainTextBlock().Text) {
				empty = false
			}
		}
	}
	return empty
}

func (r *ResponseConstructor) isEmptyString(str string) bool {
	return strings.TrimSpace(str) == ""
}

func (r *ResponseConstructor) tryExecuteTemplatesOnTrainFaq(faq *seopb.TrainDirectionFAQItem, language lang.Lang, data *models.SeoDirectionData) error {
	t, err := r.tryExecuteTemplate(faq.Title, language, data)
	if err != nil {
		return err
	}
	faq.Title = t

	err = r.tryExecuteTemplatesOnTextBlock(faq.Text, language, data)
	if err != nil {
		return err
	}
	return nil
}

func (r *ResponseConstructor) tryExecuteTemplatesOnBaseFaq(faq *seo_pages.FAQSchemaMarkupItem, language lang.Lang, data *models.SeoDirectionData) error {
	t, err := r.tryExecuteTemplate(faq.Question, language, data)
	if err != nil {
		return err
	}
	faq.Question = t

	t, err = r.tryExecuteTemplate(faq.Answer, language, data)
	if err != nil {
		return err
	}
	faq.Answer = t

	return nil
}
