package seo

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"

	seopb "a.yandex-team.ru/travel/trains/search_api/api/seo_direction"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/dict/registry"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/express"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/lang"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/points"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/schedule"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/tariffs/cache"
)

type Service struct {
	dataCollector       *DataCollector
	responseConstructor *ResponseConstructor
	keySets             *KeySetsResource
	logger              log.Logger
}

func NewService(
	durationTranslator *i18n.DurationTranslator,
	translatableFactory *i18n.TranslatableFactory,
	carTypeTranslator *i18n.CarTypeTranslator,
	tariffCache *cache.TariffCache,
	repoRegistry *registry.RepositoryRegistry,
	expressRepository *express.Repository,
	scheduleRepository *schedule.Repository,
	pointParser *points.Parser,
	keySets *KeySetsResource,
	logger log.Logger,
) *Service {
	dataCollector := NewDataCollector(
		durationTranslator,
		translatableFactory,
		carTypeTranslator,
		nil,
		tariffCache,
		repoRegistry,
		expressRepository,
		scheduleRepository,
		pointParser,
		logger,
	)
	return &Service{
		dataCollector: dataCollector,
		keySets:       keySets,
		logger:        logger,
	}
}

func (s *Service) SeoDirection(ctx context.Context, fromSlug, toSlug string, language lang.Lang) (*seopb.SeoDirectionResponse, error) {
	funcName := "seo.Service.SeoDirection"
	data, err := s.dataCollector.GetSeoDirectionsData(ctx, fromSlug, toSlug, language)
	if err != nil {
		return nil, fmt.Errorf("%s: data collecting error: %w", funcName, err)
	}
	rsp, err := s.responseConstructor.BuildSeoDirection(data, language)
	if err != nil {
		return nil, fmt.Errorf("%s: responce build error: %w", funcName, err)
	}
	return rsp, nil
}

func (s *Service) UpdateKeySets() error {
	funcName := "seo.Service.UpdateKeySets"
	commonKs := s.keySets.GetKeySet(KeySetNameCommon)
	directionKs := s.keySets.GetKeySet(KeySetNameDirection)
	seoResponseConstructor, err := NewResponseConstructor(
		commonKs,
		directionKs,
		s.logger,
	)
	if err != nil {
		return fmt.Errorf("%s: responseConstructor build failed: %w", funcName, err)
	}
	s.responseConstructor = seoResponseConstructor
	s.dataCollector.UpdateKeySet(commonKs)
	return nil
}
