package app

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	tanker "a.yandex-team.ru/travel/library/go/tanker/next"
	tankerpb "a.yandex-team.ru/travel/proto/tanker"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/geo"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/i18n"
	"a.yandex-team.ru/travel/trains/search_api/internal/pkg/templater"
	"a.yandex-team.ru/travel/trains/search_api/internal/seo"
)

type App struct {
	// common
	logger  log.Logger
	cfg     *Config
	cancels []func()

	geobaseClient geo.Geobase

	// i18n
	keyset                i18n.FileKeyset
	trainTitleTranslator  *i18n.TrainTitleTranslator
	timeTranslator        *i18n.TimeTranslator
	translatableFactory   *i18n.TranslatableFactory
	linguisticsTranslator *i18n.LinguisticsTranslator

	// seo
	tankerHTTPClient       *tanker.TankerHTTPClient
	seoResponseConstructor *seo.ResponseConstructor
	seoChecker             *seo.TemplateChecker
	seoKeySets             *seo.KeySetsResource
}

func NewApp(logger log.Logger, cfg *Config) (*App, error) {
	const funcName = "app.NewApp"
	ctx, cancelCtx := context.WithCancel(context.Background())
	a := &App{
		logger:  logger,
		cfg:     cfg,
		cancels: []func(){cancelCtx},
	}

	var err error
	defer func() {
		if err != nil {
			a.Destroy()
		}
	}()

	a.geobaseClient = geo.NewFakeGeobaseClient()
	a.cancels = append(a.cancels, a.geobaseClient.Destroy)

	err = a.buildI18n()
	if err != nil {
		return nil, xerrors.Errorf("%s: %w", funcName, err)
	}

	templater.SetTranslator(a.translatableFactory)

	err = a.buildSeo(ctx)
	if err != nil {
		return nil, xerrors.Errorf("%s: %w", funcName, err)
	}

	return a.run(ctx)
}

func (a *App) Destroy() {
	for _, cancel := range a.cancels {
		cancel()
	}
}

func (a *App) buildI18n() error {
	const funcName = "trains.search_api.internal.pkg.app.App.buildI18n"

	var err error
	a.keyset, err = i18n.ReadKeyset(a.cfg.I18n.KeysetPath)
	if err != nil {
		return fmt.Errorf("%s: keyset creation fails: %w", funcName, err)
	}
	a.trainTitleTranslator = i18n.NewTrainTitleTranslator(a.keyset)
	a.timeTranslator = i18n.NewTimeTranslator(i18n.WithKeysetPath(a.logger, &a.cfg.I18n))
	a.linguisticsTranslator = i18n.NewLinguisticsTranslator(
		a.geobaseClient, a.keyset, a.cfg.Direction.LanguageFallbacks, a.cfg.Direction.LanguageCaseFallbacks)
	a.translatableFactory = i18n.NewTranslatableFactory(a.linguisticsTranslator, a.keyset)
	return nil
}

func (a *App) buildSeo(ctx context.Context) error {
	const funcName = "trains.search_api.internal.pkg.app.App.buildSeo"
	var err error
	tc, err := tanker.NewTankerHTTPClient(
		a.cfg.TankerBaseURL,
		a.logger,
		a.cfg.TankerOAuthToken,
	)
	if err != nil {
		return xerrors.Errorf("%s: failed to create TankerHTTPClient: %w", funcName, err)
	}
	a.tankerHTTPClient = tc

	if a.cfg.TankerBranch == "" {
		return xerrors.Errorf("%s: empty TankerBranch", funcName)
	}
	ksCommon, err := tc.GetKeySetData(ctx, "travel-backend", a.cfg.TankerBranch, seo.KeySetNameCommon)
	if err != nil {
		return xerrors.Errorf("%s: failed to get %s key-set: %w", funcName, seo.KeySetNameCommon, err)
	}
	ksDirection, err := tc.GetKeySetData(ctx, "travel-backend", a.cfg.TankerBranch, seo.KeySetNameDirection)
	if err != nil {
		return xerrors.Errorf("%s: failed to get %s key-set: %w", funcName, seo.KeySetNameDirection, err)
	}
	ksDirectionTests, err := tc.GetKeySetData(ctx, "travel-backend", a.cfg.TankerBranch, seo.KeySetNameDirectionTests)
	if err != nil {
		return xerrors.Errorf("%s: failed to get %s key-set: %w", funcName, seo.KeySetNameDirectionTests, err)
	}

	tankerKeySets := &tankerpb.KeySetExportResponse{
		Keysets: map[string]*tankerpb.KeySet{
			seo.KeySetNameCommon:         ksCommon.Keysets[seo.KeySetNameCommon],
			seo.KeySetNameDirection:      ksDirection.Keysets[seo.KeySetNameDirection],
			seo.KeySetNameDirectionTests: ksDirectionTests.Keysets[seo.KeySetNameDirectionTests],
		},
	}

	a.seoKeySets, err = seo.NewResourceFromSnapshot(tankerKeySets)
	if err != nil {
		return xerrors.Errorf("%s: init key-sets failed: %w", funcName, err)
	}

	responseConstructor, err := seo.NewResponseConstructor(
		a.seoKeySets.GetKeySet(seo.KeySetNameCommon),
		a.seoKeySets.GetKeySet(seo.KeySetNameDirection),
		a.logger,
	)
	if err != nil {
		return xerrors.Errorf("%s: failed to create seoResponseConstructor: %w", funcName, err)
	}
	a.seoResponseConstructor = responseConstructor
	tester, err := seo.NewTemplateChecker(
		a.seoKeySets.GetKeySet(seo.KeySetNameDirectionTests),
		a.logger,
		a.seoResponseConstructor,
	)
	if err != nil {
		return xerrors.Errorf("%s: failed to create TemplateChecker: %w", funcName, err)
	}
	a.seoChecker = tester
	return nil
}

func (a *App) run(ctx context.Context) (*App, error) {
	const funcName = "trains.search_api.internal.pkg.app.App.run"

	var err error
	if !a.cfg.SkipChecks {
		err = a.seoChecker.RunChecks()
		if err != nil {
			return nil, xerrors.Errorf("%s: tests failed: %w", funcName, err)
		}
	}
	if !a.cfg.SkipDump {
		tankerResourceWriter := resourcestorage.NewS3StorageWriter(
			resourcestorage.S3StorageConfig(a.cfg.Seo.S3Storage),
			a.cfg.Seo.S3AccessKey,
			a.cfg.Seo.S3Secret,
		)
		dumper := resourcestorage.NewDumper(
			a.seoKeySets,
			a.cfg.Seo.SnapshotDirectory,
			tankerResourceWriter,
			2,
			a.logger,
		)
		_, err = dumper.Dump()
		if err != nil {
			return nil, xerrors.Errorf("%s: tanker key-sets dump failed: %w", funcName, err)
		}
		a.logger.Infof("%s: tanker key-sets dump success", funcName)
	}
	return a, nil
}
