package main

import (
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/jonboulle/clockwork"
	ghealth "google.golang.org/grpc/health"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/trains/worker/internal/app"
	"a.yandex-team.ru/travel/trains/worker/internal/handler"
	"a.yandex-team.ru/travel/trains/worker/internal/healthswitch"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/balancer"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/task"
)

const serviceName = "worker"

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()

	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewQloudClient()
		if err != nil {
			fmt.Println("failed to create tvm client:", err)
		}
		tvmClient, err = cachedtvm.NewClient(tvmClient)
		if err != nil {
			fmt.Println("failed to create cached tvm client:", err)
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()

	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	raspRepo := dict.NewRepo(&app.Cfg.DictRasp, logger)
	err = raspRepo.Load()
	if err != nil {
		logger.Errorf("failed to load raspRepo %s", err.Error())
		return
	}

	healthStatus := healthswitch.NewSwitch()

	connection, err := balancer.CreateConnection(app.Cfg.WorkerBalancer.YPlannerID, logger)
	if err != nil {
		logger.Fatal("failed to create connection to workerService", log.Error(err))
	}

	defer func() {
		if err := connection.Close(); err != nil {
			logger.Error("failed to close connection to workerService", log.Error(err))
		}
	}()

	grpcWorkerBalancerHandler := balancer.NewGRPCWorkerBalancerHandler(logger, rootRegistry, connection, 2*time.Second)
	grpcWorkerHandler, err := handler.NewGRPCWorkerHandler(ctx, logger, rootRegistry, &app.Cfg.TrainTariffsTask, raspRepo)

	if err != nil {
		logger.Fatal("failed to create GRPCWorkerHandler", log.Error(err))
	}

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		grpcWorkerHandler.GetServiceRegisterer(),
		grpcWorkerBalancerHandler.GetServiceRegisterer(),
	}

	healthServer := ghealth.NewServer()

	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		app.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	).Requires(
		func() bool {
			return healthStatus.GetHealth()
		},
	)
	servingStatusService.MonitorServingStatus(ctx)

	wg := sync.WaitGroup{}
	wg.Add(4)

	searchWorker, err := task.NewSearchWorker(ctx, grpcWorkerHandler.SearchTaskQueue, logger, rootRegistry, &app.Cfg.TrainTariffsTask,
		app.Cfg.LogbrokerProducer, raspRepo)

	if err != nil {
		fmt.Println("failed to create searchWorker:", err)
		return
	}

	go func() {
		defer wg.Done()
		err = searchWorker.Run(ctx)
		if err != nil {
			logger.Fatal("Error while starting searchWorker", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		server := grpcserver.NewDefaultGrpcServerBuilder(
			app.Cfg.Grpc,
			grpcServiceRegisterers,
			logger,
			tvmClient,
			tvmAllowedIds,
			rootRegistry.WithPrefix("grpc"),
		).WithHealthServer(healthServer).Build()
		err = server.Run(ctx)
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(ctx, app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
