package handler

import (
	"context"
	"strconv"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	travelMetrics "a.yandex-team.ru/travel/library/go/metrics"
	tpb "a.yandex-team.ru/travel/proto"
	tariffsModels "a.yandex-team.ru/travel/trains/library/go/tariffs/models"
	pb "a.yandex-team.ru/travel/trains/worker/api"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/task"
)

type GRPCWorkerHandler struct {
	logger          log.Logger
	SearchTaskQueue *task.SearchTaskQueue
	appMetrics      *travelMetrics.AppMetrics
	task            *task.TrainTariffsTask
}

func NewGRPCWorkerHandler(ctx context.Context, logger *zap.Logger, metricsRegistry metrics.Registry,
	taskConfig *task.TrainTariffsTaskConfig, raspRepo *dict.DictRepo) (*GRPCWorkerHandler, error) {
	appMetrics := travelMetrics.NewAppMetrics(metricsRegistry)
	trainTariffsTask, err := task.NewTrainTariffsTask(ctx, nil, logger, taskConfig, nil, nil, raspRepo)
	if err != nil {
		return nil, err
	}
	return &GRPCWorkerHandler{
		logger:          logger,
		appMetrics:      appMetrics,
		SearchTaskQueue: task.NewSearchTaskQueue(appMetrics),
		task:            trainTariffsTask,
	}, nil
}

func (h *GRPCWorkerHandler) GetTariffs(ctx context.Context, request *pb.TSegmentsRequest) (*pb.TSegmentsResponse, error) {
	h.logger.Info(request.From + " " + request.To + " " + strconv.Itoa(int(request.Date.Year)) + " " +
		strconv.Itoa(int(request.Date.Month)) + " " + strconv.Itoa(int(request.Date.Day)))
	position, err := h.SearchTaskQueue.Push(request)
	if err != nil {
		h.logger.Errorf("GRPCWorkerHandler.GetTariffs: %s", err.Error())
		return &pb.TSegmentsResponse{
			Header: &pb.TResponseHeader{
				Code: tpb.EErrorCode_EC_GENERAL_ERROR,
				Error: &tpb.TError{
					Code:    tpb.EErrorCode_EC_GENERAL_ERROR,
					Message: err.Error(),
				},
			},
		}, nil
	}

	return &pb.TSegmentsResponse{
		Header: &pb.TResponseHeader{
			Code: tpb.EErrorCode_EC_OK,
		},
		QueuePosition: uint32(position),
	}, nil
}

func (h *GRPCWorkerHandler) GetTariffsSync(ctx context.Context, request *pb.TSegmentsRequest) (*pb.TSegmentsResult, error) {
	const funcName = "GRPCWorkerHandler.GetTariffsSync"
	h.logger.Info(request.From + " " + request.To + " " + strconv.Itoa(int(request.Date.Year)) + " " +
		strconv.Itoa(int(request.Date.Month)) + " " + strconv.Itoa(int(request.Date.Day)))
	req, err := h.task.ConstructSearchTrainPricingRequest(request)
	if err != nil {
		h.logger.Errorf("%s: incorrect request", funcName)
		return buildTSegmentsResultWithErrorMessage(err.Error()), nil
	}

	directionTariffTrains, err := h.task.BuildSegmentsFromIMResponse(req, request)
	if err != nil {
		h.logger.Infof("%s: Segments not found", funcName)
		return buildEmptyTSegmentsResult(err.Error()), nil
	}

	message := new(pb.TSegmentsResult)
	if message.TariffTrain, err = tariffsModels.MapDirectionTariffTrainJSONToAPI(ctx, directionTariffTrains...); err != nil {
		h.logger.Errorf("%s: mapping json tariff failed: %v", funcName, err)
		return buildTSegmentsResultWithErrorMessage("mapping json tariff failed"), nil
	}

	return &pb.TSegmentsResult{
		Header: &pb.TResponseHeader{
			Code: tpb.EErrorCode_EC_OK,
		},
		TariffTrain: message.TariffTrain,
	}, nil
}

func buildTSegmentsResultWithErrorMessage(errorMessage string) *pb.TSegmentsResult {
	return &pb.TSegmentsResult{
		Header: &pb.TResponseHeader{
			Code: tpb.EErrorCode_EC_GENERAL_ERROR,
			Error: &tpb.TError{
				Code:    tpb.EErrorCode_EC_GENERAL_ERROR,
				Message: errorMessage,
			},
		},
		TariffTrain: nil,
	}
}

func buildEmptyTSegmentsResult(errorMessage string) *pb.TSegmentsResult {
	return &pb.TSegmentsResult{
		Header: &pb.TResponseHeader{
			Code: tpb.EErrorCode_EC_NOT_FOUND,
			Error: &tpb.TError{
				Code:    tpb.EErrorCode_EC_NOT_FOUND,
				Message: errorMessage,
			},
		},
		TariffTrain: nil,
	}
}

func (h *GRPCWorkerHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterWorkerServiceServer(server, h)
	}
}
