package balancer

import (
	"context"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/balancer"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	hcbalancer "a.yandex-team.ru/travel/library/go/grpcutil/client/healthcheck_balancer"
	travelMetrics "a.yandex-team.ru/travel/library/go/metrics"
	pb "a.yandex-team.ru/travel/trains/worker/api"
)

type GRPCWorkerBalancerHandler struct {
	logger     log.Logger
	appMetrics *travelMetrics.AppMetrics
	balancer   Balancer
}

func NewGRPCWorkerBalancerHandler(logger log.Logger, metricsRegistry metrics.Registry, connection *grpc.ClientConn,
	timeout time.Duration) *GRPCWorkerBalancerHandler {

	workerHealthCheckServiceName := "worker"
	balancerLogger, _ := zap.NewDeployLogger(log.DebugLevel)
	balancer.Register(
		hcbalancer.NewBalancerBuilder(
			hcbalancer.WithLogger(balancerLogger.WithName("WorkerBalancer")),
			hcbalancer.WithHealthCheckServiceName(workerHealthCheckServiceName),
			hcbalancer.WithHealthCheckInterval(10*time.Second),
			hcbalancer.WithBalancingMethod(hcbalancer.BalancingMethodChooseLexicographicallyFirst),
		),
	)

	appMetrics := travelMetrics.NewAppMetrics(metricsRegistry)

	return &GRPCWorkerBalancerHandler{
		logger:     logger,
		appMetrics: appMetrics,
		balancer:   *NewBalancer(logger, connection, timeout),
	}
}

func (h *GRPCWorkerBalancerHandler) GetTariffs(ctx context.Context, request *pb.TSegmentsRequest) (*pb.TSegmentsResult, error) {
	return h.balancer.client.GetTariffsSync(ctx, request)
}

func (h *GRPCWorkerBalancerHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterWorkerBalancerServiceServer(server, h)
	}
}

type Balancer struct {
	logger  log.Logger
	timeout time.Duration
	client  pb.WorkerServiceClient
}

func NewBalancer(logger log.Logger, connection *grpc.ClientConn, timeout time.Duration) *Balancer {
	grpcClient := pb.NewWorkerServiceClient(connection)
	return &Balancer{
		logger:  logger,
		client:  grpcClient,
		timeout: timeout,
	}
}
