package imclient

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil/client"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient/models"
)

type IMClient struct {
	logger     log.Logger
	httpClient *client.HTTPClient
	path       string
}

type IMBaseAuthProvider struct {
	pos          string
	authProvider *client.HTTPBaseAuthProvider
}

func NewIMBaseAuthProvider(login string, password string, pos string) (*IMBaseAuthProvider, error) {
	const funcName = "NewIMBaseAuthProvider"
	authProvider, err := client.NewHTTPBaseAuthProvider(login, password)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}
	return &IMBaseAuthProvider{
		pos:          pos,
		authProvider: authProvider,
	}, nil
}

func (p IMBaseAuthProvider) Authentificate(r *http.Request) error {
	const funcName = "IMBaseAuthProvider.Authentificate"
	err := p.authProvider.Authentificate(r)
	if err != nil {
		return fmt.Errorf("%s: can not apply authentication data to request: %w", funcName, err)
	}
	r.Header.Set("POS", p.pos)
	return nil
}

func NewIMClient(imclientBaseURL string, imClientPath string, timeout time.Duration, logger log.Logger, login string, password string, pos string) (*IMClient, error) {
	const funcName = "NewIMClient"
	authProvider, err := NewIMBaseAuthProvider(login, password, pos)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	httpClient, err := client.NewHTTPClient(imclientBaseURL, timeout, 0, 0,
		client.ContentTypeJSON, logger, authProvider)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &IMClient{
		logger:     logger,
		httpClient: httpClient,
		path:       imClientPath,
	}, nil
}

func (c *IMClient) SearchTrainPricing(ctx context.Context, origin string, destination string, departureDate string,
	timeFrom int, timeTo int, getByLocalTime bool) (*models.SearchTrainPricingResponse, error) {
	const funcName = "IMClient.SearchTrainPricing"
	request := models.SearchTrainPricingRequest{
		Origin:         origin,
		Destination:    destination,
		DepartureDate:  departureDate,
		TimeFrom:       timeFrom,
		TimeTo:         timeTo,
		GetByLocalTime: getByLocalTime,
	}
	var response models.SearchTrainPricingResponse

	err := c.httpClient.Post(ctx, c.path+"/Railway/V1/Search/TrainPricing", request, &response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetSearchTrainRoute(ctx context.Context, trainNumber string, origin string, destination string,
	departureDate string, provider string) (*models.SearchTrainRouteResponse, error) {
	const funcName = "IMClient.GetSearchTrainRoute"
	request := models.SearchTrainRouteRequest{
		TrainNumber:   trainNumber,
		Origin:        origin,
		Destination:   destination,
		DepartureDate: departureDate,
		Provider:      provider,
	}

	var response models.SearchTrainRouteResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Search/TrainRoute", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetSearchSchedule(ctx context.Context, origin string, destination string, departureDate interface{},
	timeFrom int, timeTo int, provider string) (*models.SearchScheduleResponse, error) {
	const funcName = "IMClient.GetSearchSchedule"
	request := models.SearchScheduleRequest{
		Origin:        origin,
		Destination:   destination,
		DepartureDate: departureDate,
		TimeFrom:      timeFrom,
		TimeTo:        timeTo,
		Provider:      provider,
	}

	var response models.SearchScheduleResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Search/Schedule", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetReservationElectronicRegistration(ctx context.Context, orderItemID int, orderItemBlankIds []int, set bool,
) (*models.ReservationElectronicRegistrationResponse, error) {
	const funcName = "IMClient.GetReservationElectronicRegistration"
	request := models.ReservationElectronicRegistrationRequest{
		OrderItemID:       orderItemID,
		OrderItemBlankIds: orderItemBlankIds,
		Set:               set,
	}

	var response models.ReservationElectronicRegistrationResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Reservation/ElectronicRegistration", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetReservationUpdateBlank(ctx context.Context, orderItemID int) (*models.ReservationUpdateBlanksResponse, error) {
	const funcName = "IMClient.GetReservationUpdateBlank"
	request := models.ReservationUpdateBlanksRequest{
		OrderItemID: orderItemID,
	}

	var response models.ReservationUpdateBlanksResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Reservation/UpdateBlanks", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetSearchCarPricing(ctx context.Context, originCode string, destinationCode string, departureDate string,
	trainNumber string, carType interface{}, tariffType string, specialPlacesDemand string, getOnlyCarTransportationCoaches bool,
	getOnlyNonRefundableTariffs bool, provider string) (*models.SearchCarPricingResponse, error) {
	const funcName = "IMClient.GetSearchCarPricing"
	request := models.SearchCarPricingRequest{
		OriginCode:                      originCode,
		DestinationCode:                 destinationCode,
		DepartureDate:                   departureDate,
		TrainNumber:                     trainNumber,
		CarType:                         carType,
		TariffType:                      tariffType,
		SpecialPlacesDemand:             specialPlacesDemand,
		GetOnlyCarTransportationCoaches: getOnlyCarTransportationCoaches,
		GetOnlyNonRefundableTariffs:     getOnlyNonRefundableTariffs,
		Provider:                        provider,
	}

	var response models.SearchCarPricingResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Search/CarPricing", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetReservationBlank(ctx context.Context, orderID int, orderItemID interface{}, orderItemIds []int,
	retrieveMainServices bool, retrieveUpsales bool, blankLanguage string,
) (*models.ReservationBlankResponse, error) {
	const funcName = "IMClient.GetReservationBlank"
	request := models.ReservationBlankRequest{
		OrderID:              orderID,
		OrderItemID:          orderItemID,
		OrderItemIds:         orderItemIds,
		RetrieveMainServices: retrieveMainServices,
		RetrieveUpsales:      retrieveUpsales,
		BlankLanguage:        blankLanguage,
	}

	var response models.ReservationBlankResponse
	err := c.httpClient.Get(ctx, "/Order/V1/Reservation/Blank", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}

func (c *IMClient) GetReservationBlankAsHTML(ctx context.Context, orderItemID int, orderItemIds []int,
	orderItemBlankID interface{}, blankLanguage string) (*models.ReservationBlankAsHTMLResponse, error) {
	const funcName = "IMClient.GetReservationEBlankAsHtml"
	request := models.ReservationBlankAsHTMLRequest{
		OrderItemID:      orderItemID,
		OrderItemIds:     orderItemIds,
		OrderItemBlankID: orderItemBlankID,
		BlankLanguage:    blankLanguage,
	}

	var response models.ReservationBlankAsHTMLResponse
	err := c.httpClient.Get(ctx, "/Railway/V1/Reservation/BlankAsHtml", request, response)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	}

	return &response, nil
}
