package logbroker

import (
	"fmt"
	"math/rand"
	"os"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/recipe"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/vault"
)

const (
	defaultEndpoint = "logbroker.yandex.net"
	defaultLogLevel = log.InfoLevel
)

type ProducerConfig struct {
	OAuthTokenSecret string `config:"logbroker-oauthtokensecret,required"`
	OAuthTokenKey    string `config:"logbroker-oauthtokenkey,required"`
	Endpoint         string `config:"logbroker-endpoint, required"`
	Topic            string `config:"logbroker-topicprefix, required"`
	SourceID         string `config:"logbroker-sourceid, required"`
	TestEnv          *recipe.Env
	LogLevel         log.Level
}

var DefaultProducerConfig = ProducerConfig{
	Endpoint: defaultEndpoint,
	SourceID: makeSourceID(),
	LogLevel: defaultLogLevel,
}

type ConsumerConfig struct {
	OAuthTokenSecret     string   `config:"logbrokerconsumer-oauthtokensecret,required"`
	OAuthTokenKey        string   `config:"logbrokerconsumer-oauthtokenkey,required"`
	Endpoint             string   `config:"logbrokerconsumer-endpoint, required"`
	Clusters             []string `config:"logbrokerconsumer-clusters, required"`
	Consumer             string   `config:"logbrokerconsumer-consumer, required"`
	TopicPrefix          string   `config:"logbrokerconsumer-topicprefix, required"`
	SourceID             string   `config:"logbrokerconsumer-sourceid, required"`
	TestEnv              *recipe.Env
	ReconnectWait        time.Duration `config:"logbroker-reconnectwait, required"`
	LogLevel             log.Level
	SearchYtLockPrefix   string `config:"logbrokerconsumer-searchytlockprefix, required"`
	SegmentsYtLockPrefix string `config:"logbrokerconsumer-segmentsytlockprefix, required"`
	YtLockPoolSize       uint   `config:"logbrokerconsumer-ytlockpoolsize, required"`
}

func makeSourceID() string {
	sourceID, ok := os.LookupEnv("DEPLOY_POD_ID")
	if ok {
		return sourceID
	}
	return fmt.Sprintf("NO_DEPLOY_%d", rand.Int31())
}

func NewProducer(cfg ProducerConfig, secretResolver *vault.YavSecretsResolver, logger *zap.Logger) (*logbroker.Producer, error) {

	if cfg.TestEnv != nil {
		return logbroker.NewProducerWithRecipe(cfg.TestEnv, "", cfg.Topic, logger), nil
	}

	return logbroker.NewProducer(
		cfg.Topic,
		cfg.Endpoint,
		cfg.SourceID,
		logbroker.NewYavCredentialsProvider(
			cfg.OAuthTokenKey,
			cfg.OAuthTokenSecret,
			secretResolver),
		logger,
	)
}
