package task

import (
	"fmt"
	"math"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/api"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient/models"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
)

var trainNumberRe = regexp.MustCompile(`(\d+)(\D*)$`)
var DASH = fmt.Sprintf(" %s ", "\u2014")

func fixTrainNumber(trainNumber string) string {
	trainNumber = strings.Replace(trainNumber, "*", "", -1)
	matched := trainNumberRe.MatchString(trainNumber)
	if !matched {
		return trainNumber
	}
	numberAsString, letter := getNumberAndFirstLetter(trainNumber)
	number, err := strconv.Atoi(numberAsString)
	if err != nil {
		return numberAsString + letter
	}
	return fmt.Sprintf("%03d%s", number, letter)
}

func getNumberAndFirstLetter(trainNumber string) (string, string) {
	groups := trainNumberRe.FindStringSubmatch(trainNumber)
	var letter string
	for _, rune_ := range groups[2] {
		letter = string(rune_)
		break
	}
	return groups[1], letter
}

type coachType string
type placeReservationType string

const (
	unknownCoachType                coachType            = "unknown"
	softCoachType                   coachType            = "soft"
	platzkarteCoachType             coachType            = "platzkarte"
	compartmentCoachType            coachType            = "compartment"
	suiteCoachType                  coachType            = "suite"
	commonCoachType                 coachType            = "common"
	sittingCoachType                coachType            = "sitting"
	unknownReservationType          placeReservationType = "unknown"             // неизвестное
	usualReservationType            placeReservationType = "usual"               // выкуп мест по одному
	twoPlacesAtOnceReservationType  placeReservationType = "two_places_at_once"  // выкуп двух мест сразу - "мягкие" вагоны, часть СВ в Стрижах
	fourPlacesAtOnceReservationType placeReservationType = "four_places_at_once" // выкуп четырех мест сразу - купе-переговорные в Сапсанах
)

var coachTypeFromImCarType = map[string]coachType{
	"ReservedSeat": platzkarteCoachType,
	"Compartment":  compartmentCoachType,
	"Luxury":       suiteCoachType,
	"Shared":       commonCoachType,
	"Sedentary":    sittingCoachType,
	"Soft":         softCoachType,
}

var imPlaceReservationTypes = map[string]placeReservationType{
	"Usual":            usualReservationType,
	"TwoPlacesAtOnce":  twoPlacesAtOnceReservationType,
	"FourPlacesAtOnce": fourPlacesAtOnceReservationType,
}

type tariffError int

const (
	unknownTariffError tariffError = iota
	soldOutTariffError
	transitDocumentRequiredTariffError
	notAvailableInWebTariffError
	featureNotAllowedTariffError
	serviceNotAllowedTariffError
	carrierNotAllowedForSalesTariffError
	otherReasonOfInaccessibilityTariffError
	unsupportedReservationTypeTariffError
	tooCheapTariffError
	unsupportedCoachTypeTariffError
	childTariffError
)

var availabilityIndicationToTariffError = map[availabilityIndicationType]tariffError{
	notAvailableInWeb:            notAvailableInWebTariffError,
	featureNotAllowed:            featureNotAllowedTariffError,
	serviceNotAllowed:            serviceNotAllowedTariffError,
	carrierNotAllowedForSales:    carrierNotAllowedForSalesTariffError,
	otherReasonOfInaccessibility: otherReasonOfInaccessibilityTariffError,
}

func Max(x, y int) int {
	if x < y {
		return y
	}
	return x
}

func updateBestTariff(tariff1 *trainTariff, tariff2 *trainTariff) *trainTariff {
	var severalPrices bool
	var ticketPrice Price
	var maxServiceTariff *trainTariff
	var servicePrice Price
	var serviceClass *string
	var minTariff *trainTariff
	if tariff1.ticketPrice.value == tariff2.ticketPrice.value {
		severalPrices = tariff1.severalPrices || tariff2.severalPrices
		ticketPrice = *tariff1.ticketPrice
		// Не начисляем комиссию на белье в плацкарте.
		// Поэтому с учетом комиссии при равных ticket_price более дешевым будет билет с бОльшим service_price
		if tariff1.servicePrice.value >= tariff2.servicePrice.value {
			maxServiceTariff = tariff1
		} else {
			maxServiceTariff = tariff2
		}
		servicePrice = *maxServiceTariff.servicePrice
		serviceClass = maxServiceTariff.serviceClass
	} else {
		if tariff1.ticketPrice.value < tariff2.ticketPrice.value {
			minTariff = tariff1
		} else {
			minTariff = tariff2
		}
		severalPrices = true
		ticketPrice = *minTariff.ticketPrice
		servicePrice = *minTariff.servicePrice
		serviceClass = minTariff.serviceClass
	}

	bestTrainTariff := trainTariff{
		coachType:                 tariff1.coachType,
		ticketPrice:               &ticketPrice,
		servicePrice:              &servicePrice,
		serviceClass:              serviceClass,
		seats:                     tariff1.seats + tariff2.seats,
		lowerSeats:                tariff1.lowerSeats + tariff2.lowerSeats,
		upperSeats:                tariff1.upperSeats + tariff2.upperSeats,
		lowerSideSeats:            tariff1.lowerSideSeats + tariff2.lowerSideSeats,
		upperSideSeats:            tariff1.upperSideSeats + tariff2.upperSideSeats,
		maxSeatsInTheSameCar:      Max(tariff1.maxSeatsInTheSameCar, tariff2.maxSeatsInTheSameCar),
		severalPrices:             severalPrices,
		placeReservationType:      usualReservationType,
		isTransitDocumentRequired: false,
		availabilityIndication:    available,
		hasNonRefundableTariff:    tariff1.hasNonRefundableTariff || tariff2.hasNonRefundableTariff,
	}
	return &bestTrainTariff
}

func validate(tariff trainTariff) (bool, map[tariffError]bool) {
	errors := make(map[tariffError]bool)
	if tariff.coachType == unknownCoachType {
		errors[unsupportedCoachTypeTariffError] = true
	}
	if tariff.ticketPrice.value <= 10 {
		errors[tooCheapTariffError] = true
	}
	if tariff.placeReservationType == unknownReservationType {
		errors[unsupportedReservationTypeTariffError] = true
	}
	if tariff.seats <= 0 {
		errors[soldOutTariffError] = true
	}
	if tariff.isTransitDocumentRequired {
		errors[transitDocumentRequiredTariffError] = true
	}
	if tariff.categoryTraits["is_for_children"] {
		errors[childTariffError] = true
	}
	if tariff.availabilityIndication != available {
		tariffError, ok := availabilityIndicationToTariffError[tariff.availabilityIndication]
		if !ok {
			tariffError = unknownTariffError
		}
		errors[tariffError] = true
	}
	return len(errors) == 0, errors
}

func buildTrainTariffsClasses(trainTariffs []*trainTariff) (map[coachType]trainTariff, map[coachType][]tariffError) {
	tariffClasses := map[coachType]trainTariff{}
	brokenTariffClasses := map[coachType][]tariffError{}
	var currentGroup []*trainTariff
	for i := 0; i < len(trainTariffs); i++ {
		currentGroup = append(currentGroup, trainTariffs[i])
		if i != len(trainTariffs)-1 && trainTariffs[i+1].coachType == trainTariffs[i].coachType {
			continue
		}
		tariffClassesErrors := make(map[tariffError]bool)
		var validTariffs []*trainTariff
		for j := 0; j < len(currentGroup); j++ {
			valid, errors := validate(*currentGroup[j])
			for errorType, value := range errors {
				if value {
					tariffClassesErrors[errorType] = value
				}
			}
			if valid {
				validTariffs = append(validTariffs, currentGroup[j])
			}
		}
		if len(validTariffs) == 0 {
			var errors []tariffError
			for errorType, value := range tariffClassesErrors {
				if value {
					errors = append(errors, errorType)
				}
			}
			brokenTariffClasses[trainTariffs[i].coachType] = errors
		} else {
			bestTariff := validTariffs[0]
			for j := 1; j < len(validTariffs); j++ {
				bestTariff = updateBestTariff(bestTariff, validTariffs[j])
			}
			tariffClasses[trainTariffs[i].coachType] = *bestTariff
		}
		currentGroup = nil
	}
	return tariffClasses, brokenTariffClasses
}

func fixBrokenClasses(brokenClasses map[coachType][]tariffError) map[coachType][]tariffError {
	if len(brokenClasses) == 0 {
		return brokenClasses
	}
	if _, ok := brokenClasses[unknownCoachType]; ok {
		var errors []tariffError
		for i := 0; i < len(brokenClasses[unknownCoachType]); i++ {
			if brokenClasses[unknownCoachType][i] != tooCheapTariffError &&
				brokenClasses[unknownCoachType][i] != unsupportedCoachTypeTariffError &&
				brokenClasses[unknownCoachType][i] != serviceNotAllowedTariffError {
				errors = append(errors, brokenClasses[unknownCoachType][i])
			}
		}
		brokenClasses[unknownCoachType] = errors
		if len(brokenClasses[unknownCoachType]) == 0 {
			delete(brokenClasses, unknownCoachType)
		}
	}
	return brokenClasses
}

const (
	priceCurrency  = pb.ECurrency_C_RUB
	pricePrecision = 2
)

func priceToProto(price Price) *pb.TPrice {
	if price.currency != "RUB" && price.currency != "RUR" {
		return nil
	}
	return &pb.TPrice{
		Currency:  priceCurrency,
		Precision: pricePrecision,
		Amount:    int64(price.value * math.Pow10(pricePrecision)),
	}
}

func minTariffToProto(minTariff *trainTariff) *api.MinTariffsClass {
	return &api.MinTariffsClass{
		Price: priceToProto(*minTariff.ticketPrice),
		Seats: uint32(minTariff.seats),
	}
}

func extractBrokenClassesCode(brokenClasses map[coachType][]tariffError) (api.BrokenClassesCode, bool) {
	hasOtherCode := false
	for _, codes := range [][]tariffError{
		brokenClasses[commonCoachType],
		brokenClasses[compartmentCoachType],
		brokenClasses[platzkarteCoachType],
		brokenClasses[sittingCoachType],
		brokenClasses[softCoachType],
		brokenClasses[suiteCoachType],
		brokenClasses[unknownCoachType],
	} {
		for _, code := range codes {
			if code == soldOutTariffError {
				return api.BrokenClassesCode_BROKEN_CLASSES_CODE_SOLD_OUT, true
			}
			hasOtherCode = true
		}
	}
	if hasOtherCode {
		return api.BrokenClassesCode_BROKEN_CLASSES_CODE_OTHER, true
	}
	return api.BrokenClassesCode_BROKEN_CLASSES_CODE_INVALID, false
}

func buildSettlement(raspRepo *dict.DictRepo, station *rasp.TStation) (*api.Settlement, error) {
	settlement, ok := raspRepo.GetSettlement(station.SettlementId)
	if !ok {
		return nil, fmt.Errorf("not found settlement by settlementId")
	}
	return &api.Settlement{
		Id: uint32(settlement.Id),
		//Preposition: settlement.Title.Ru.Prepositional,
		Title:           settlement.Title.Ru.Nominative,
		TitleAccusative: settlement.Title.Ru.Accusative,
		TitleGenitive:   settlement.Title.Ru.Genitive,
		//TitleLocative: settlement.Title.,
	}, nil
}

const expressFakeID = 100

func buildStation(raspRepo *dict.DictRepo, stationCode int) (*api.Station, error) {
	const funcName = "buildStation"
	station, ok := raspRepo.GetStationByExpressCode(int32(stationCode))
	if !ok {
		return nil, fmt.Errorf("not found station by express code")
	}
	title := station.Title.Ru
	stationID := fmt.Sprintf("%d", station.Id)

	originTz, ok := raspRepo.GetTimeZone(station.TimeZoneId)
	if !ok {
		return nil, fmt.Errorf("not found timezone by station.timeZoneID")
	}
	tz := originTz.Code

	settlement, err := buildSettlement(raspRepo, station)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}

	return &api.Station{
		Id:              stationID,
		Title:           title,
		Timezone:        tz,
		RailwayTimezone: tz,
		Country: &api.Country{
			Id: station.CountryId,
			//code : "RU",
		},
		Settlement: settlement,
	}, nil
}

func getTime(raspRepo *dict.DictRepo, stringDatetime string, stationCode int) (*time.Time, error) {
	layout := "2006-01-02T15:04:05"
	t, err := time.Parse(layout, stringDatetime)
	if err != nil {
		return nil, fmt.Errorf("can not parsing datetime: %w", err)
	}

	station, ok := raspRepo.GetStationByExpressCode(int32(stationCode))
	if !ok {
		return nil, fmt.Errorf("not found station by express code: %d", stationCode)
	}

	tz, ok := raspRepo.GetTimeZone(station.RailwayTimeZoneId)
	if !ok {
		return nil, fmt.Errorf("not found timezone by railwayTimezoneID code: %d", station.RailwayTimeZoneId)
	}

	location, err := time.LoadLocation(tz.Code)
	if err != nil {
		return nil, fmt.Errorf("can not load location: %w", err)
	}

	timeInZone := time.Date(t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute(), t.Second(), 0, location)
	utc, _ := time.LoadLocation("UTC")
	ansTime := timeInZone.In(utc)

	return &ansTime, nil
}

func getTimeInZone(raspRepo *dict.DictRepo, stringDatetime string, stationCode int) (*time.Time, error) {
	layout := "2006-01-02T15:04:05"
	t, err := time.Parse(layout, stringDatetime)
	if err != nil {
		return nil, fmt.Errorf("can not parsing datetime: %w", err)
	}

	station, ok := raspRepo.GetStationByExpressCode(int32(stationCode))
	if !ok {
		return nil, fmt.Errorf("not found station by express code: %d", stationCode)
	}

	tz, ok := raspRepo.GetTimeZone(station.RailwayTimeZoneId)
	if !ok {
		return nil, fmt.Errorf("not found timezone by railwayTimezoneID code: %d", station.RailwayTimeZoneId)
	}

	location, err := time.LoadLocation(tz.Code)
	if err != nil {
		return nil, fmt.Errorf("can not load location: %w", err)
	}

	timeInZone := time.Date(t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute(), t.Second(), 0, location)
	//ansTime := t.In(location)

	return &timeInZone, nil
}

func GetTitlePartsFromThread(raspRepo *dict.DictRepo, trainNumber string) ([]*rasp.TThreadTitlePart, error) {
	ans, ok := raspRepo.GetThreadByTrainNumber(trainNumber)

	if !ok {
		return nil, fmt.Errorf("not found thread with trainNumber: %s", trainNumber)
	}
	var parts []*rasp.TThreadTitlePart
	if ans.CommonTitle != nil && ans.TransportType == rasp.TTransport_TYPE_TRAIN {
		for i := 0; i < len(ans.CommonTitle.TitleParts); i++ {
			parts = append(parts, ans.CommonTitle.TitleParts[i])
		}
		return parts, nil
	}
	return nil, fmt.Errorf("can not get stations from thread")
}

func checkDurationAndTimezones(train models.Train, railwayDeparture time.Time, railwayArrival time.Time, logger log.Logger) {
	duration := train.TripDuration
	if (railwayArrival.Unix()-railwayDeparture.Unix())/60 != int64(duration) {
		logger.Warnf("Рассчитанное и указанное время прибытия отличаются %s %s", railwayDeparture, railwayArrival)
	}
}

func fillInplaceSegmentTimeAndStations(raspRepo *dict.DictRepo, segment *TrainSegment, train models.Train,
	request *models.SearchTrainPricingRequest, logger log.Logger) {
	origin, _ := strconv.Atoi(request.Origin)
	destination, _ := strconv.Atoi(request.Destination)
	railwayDeparture, _ := getTimeInZone(raspRepo, train.DepartureDateTime, origin)
	railwayArrival, _ := getTimeInZone(raspRepo, train.ArrivalDateTime, destination)

	checkDurationAndTimezones(train, *railwayDeparture, *railwayArrival, logger)

	segment.stationFromExpressCode = train.OriginStationCode
	segment.stationToExpressCode = train.DestinationStationCode

	segment.startExpressTitleOrCode = train.OriginName
	segment.endExpressTitleOrCode = train.DestinationName
	segment.railwayDeparture = railwayDeparture
	segment.railwayArrival = railwayArrival
	segment.departure, _ = getTime(raspRepo, train.DepartureDateTime, origin)
	segment.arrival, _ = getTime(raspRepo, train.ArrivalDateTime, destination)
	segment.isDeluxe = strings.Contains(train.TrainDescription, "ФИРМ")
}

func makeTariffSegmentKeyExact(number string, departure *time.Time) string {
	var key string
	if true { //settings.ALIGN_SEARCH_SEGMENT_KEYS
		key = makeDatetimeKey(*departure)
	} else {
		key = makePreciseDatetimeKey(*departure)
	}
	return fmt.Sprintf("train %s %s", number, key)
}

func makeTariffSegmentKey(segment TrainSegment) string {
	return makeTariffSegmentKeyExact(segment.originalNumber, segment.railwayDeparture) //departure)
}

func getExpressStation(pointCode string, raspRepo *dict.DictRepo) (*rasp.TStation, error) {
	numberPointCode, _ := strconv.Atoi(pointCode)
	station, ok := raspRepo.GetStationByExpressCode(int32(numberPointCode))

	if !ok {
		return nil, fmt.Errorf("not found station by pointCode %d", numberPointCode)
	}
	return station, nil
}

func BuildTrainSegments(response *models.SearchTrainPricingResponse, request *models.SearchTrainPricingRequest,
	raspRepo *dict.DictRepo, logger log.Logger) []*TrainSegment {
	var segments []*TrainSegment

	for _, train := range response.Trains {
		if train.TransportType != "Train" {
			continue
		}

		segment := TrainSegment{}
		segment.rawTrainCategory = strings.TrimSpace(train.TrainDescription)
		segment.rawTrainName = strings.TrimSpace(train.TrainName)
		segment.originalNumber = fixTrainNumber(train.TrainNumber)
		segment.number = fixTrainNumber(train.DisplayTrainNumber)
		if segment.number == "" {
			segment.number = segment.originalNumber
		}

		segment.trainNumberToGetRoute = train.TrainNumberToGetRoute

		segment.possibleNumbers = getPossibleNumbers(segment.originalNumber)
		segment.tariffs = make(map[string]bool)
		segment.tariffs["electronic_ticket"] = train.HasElectronicRegistration
		segment.canSupplySegments = true

		elems := append([]string{train.OriginName}, train.DestinationNames...)
		segment.ufsTitle = strings.Join(elems, DASH)

		segment.coachOwners = train.Carriers
		segment.hasDynamicPricing = train.HasDynamicPricingCars
		segment.twoStorey = train.HasTwoStoreyCars
		segment.isSuburban = train.IsSuburban
		segment.provider = train.Provider

		fillInplaceSegmentTimeAndStations(raspRepo, &segment, train, request, logger)

		segment.key = makeTariffSegmentKey(segment)

		var trainTariffs []*trainTariff
		for j := 0; j < len(train.CarGroups); j++ {
			trainTariffs = append(trainTariffs, parseTrainTariff(&train.CarGroups[j]))
		}

		sort.Slice(trainTariffs, func(i, j int) bool {
			return trainTariffs[i].coachType < trainTariffs[j].coachType
		})
		classes, brokenClasses := buildTrainTariffsClasses(trainTariffs)

		brokenClasses = fixBrokenClasses(brokenClasses)

		segment.apiSegment = &api.Segment{}

		segment.apiSegment.Tariffs = &api.MinTariffs{
			Classes: &api.MinTariffsClasses{
				Soft: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
				Platzkarte: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
				Compartment: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
				Suite: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
				Common: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
				Sitting: &api.MinTariffsClass{
					Price: &pb.TPrice{},
				},
			},
		}
		segment.brokenClasses = brokenClasses

		if brokenClassesCode, ok := extractBrokenClassesCode(brokenClasses); ok {
			segment.apiSegment.Tariffs.OptionalBrokenClassesCode = &api.MinTariffs_BrokenClassesCode{BrokenClassesCode: brokenClassesCode}
		}

		segment.tariffsClasses = make(map[coachType]trainTariff)
		for class, value := range classes {
			segment.tariffsClasses[class] = value
			switch class {
			case "soft":
				segment.apiSegment.Tariffs.Classes.Soft = minTariffToProto(&value)
			case "platzkarte":
				segment.apiSegment.Tariffs.Classes.Platzkarte = minTariffToProto(&value)
			case "compartment":
				segment.apiSegment.Tariffs.Classes.Compartment = minTariffToProto(&value)
			case "suite":
				segment.apiSegment.Tariffs.Classes.Suite = minTariffToProto(&value)
			case "common":
				segment.apiSegment.Tariffs.Classes.Common = minTariffToProto(&value)
			case "sitting":
				segment.apiSegment.Tariffs.Classes.Sitting = minTariffToProto(&value)
			}
		}
		segments = append(segments, &segment)
	}

	return segments
}

func parseCarDescriptions(carDescriptions []string) map[string]bool {
	var categoryTraits = map[string]bool{
		"can_choose_male_female_mix_places": false,
		"places_without_numbers":            false,
		"is_not_firm":                       false,
		"pet_in_coach":                      false,
		"pet_places_only":                   false,
		"is_for_children":                   false,
	}
	for i := 0; i < len(carDescriptions); i++ {
		types := strings.Split(strings.Replace(carDescriptions[i], "* ", "*", -1), " ")
		for j := 0; j < len(types); j++ {
			switch types[j] {
			case "МЖ":
				categoryTraits["can_choose_male_female_mix_places"] = true
			case "БН":
				categoryTraits["places_without_numbers"] = true
			case "НФ":
				categoryTraits["is_not_firm"] = true
			case "Ж":
				categoryTraits["pet_in_coach"] = true
			case "*Ж":
				categoryTraits["pet_places_only"] = true
			case "*Д":
				categoryTraits["is_for_children"] = true
			}
		}
	}
	return categoryTraits
}

type Price struct {
	value    float64
	currency string
}

type trainTariff struct {
	coachType                 coachType
	ticketPrice               *Price
	servicePrice              *Price
	severalPrices             bool
	seats                     int
	lowerSeats                int
	lowerSideSeats            int
	upperSeats                int
	upperSideSeats            int
	maxSeatsInTheSameCar      int
	placeReservationType      placeReservationType
	isTransitDocumentRequired bool
	availabilityIndication    availabilityIndicationType
	serviceClass              *string
	categoryTraits            map[string]bool
	fee                       *Price
	feePercent                float64
	orderURL                  string
	hasNonRefundableTariff    bool
}

type availabilityIndicationType int

const (
	unknownAvailabilityIndication availabilityIndicationType = iota
	available
	notAvailableInWeb
	featureNotAllowed
	serviceNotAllowed
	carrierNotAllowedForSales
	otherReasonOfInaccessibility
)

var imAvailabilityIndicationTypes = map[string]availabilityIndicationType{
	"Available":                    available,
	"NotAvailableInWeb":            notAvailableInWeb,
	"FeatureNotAllowed":            featureNotAllowed,
	"ServiceNotAllowed":            serviceNotAllowed,
	"CarrierNotAllowedForSales":    carrierNotAllowedForSales,
	"OtherReasonOfInaccessibility": otherReasonOfInaccessibility,
}

func parseTrainTariff(imCoachGroup *models.CarGroup) *trainTariff {
	imCarType := imCoachGroup.CarType
	coachType, ok := coachTypeFromImCarType[imCarType]
	if !ok {
		coachType = unknownCoachType
	}

	placeReservationType := parsePlaceReservationTypes(imCoachGroup)

	minTariff := imCoachGroup.MinPrice
	maxTariff := imCoachGroup.MaxPrice
	serviceTariff := imCoachGroup.ServiceCosts[0]
	hasNonRefundableTariff := imCoachGroup.HasNonRefundableTariff

	seats := imCoachGroup.TotalPlaceQuantity
	lowerSeats := imCoachGroup.LowerPlaceQuantity
	lowerSideSeats := imCoachGroup.LowerSidePlaceQuantity
	upperSeats := imCoachGroup.UpperPlaceQuantity
	upperSideSeats := imCoachGroup.UpperSidePlaceQuantity
	severalPrices := maxTariff != minTariff
	var serviceClass *string = nil
	if len(imCoachGroup.ServiceClasses) != 0 {
		serviceClass = &imCoachGroup.ServiceClasses[0]
	}

	categoryTraits := parseCarDescriptions(imCoachGroup.CarDescriptions)

	availabilityIndication, ok := imAvailabilityIndicationTypes[imCoachGroup.AvailabilityIndication]
	if !ok {
		availabilityIndication = unknownAvailabilityIndication
	}

	return &trainTariff{
		coachType: coachType,
		ticketPrice: &Price{
			value:    minTariff,
			currency: "RUB",
		},
		servicePrice: &Price{
			value:    serviceTariff,
			currency: "RUB",
		},
		severalPrices:             severalPrices,
		seats:                     seats,
		lowerSeats:                lowerSeats,
		lowerSideSeats:            lowerSideSeats,
		upperSeats:                upperSeats,
		upperSideSeats:            upperSideSeats,
		maxSeatsInTheSameCar:      seats,
		placeReservationType:      placeReservationType,
		isTransitDocumentRequired: imCoachGroup.IsTransitDocumentRequired,
		availabilityIndication:    availabilityIndication,
		serviceClass:              serviceClass,
		categoryTraits:            categoryTraits,
		hasNonRefundableTariff:    hasNonRefundableTariff,
	}
}

func parsePlaceReservationTypes(imCoachGroup *models.CarGroup) placeReservationType {
	imPlaceReservationType := imCoachGroup.PlaceReservationTypes[0]
	reservationType, ok := imPlaceReservationTypes[imPlaceReservationType]
	if !ok {
		reservationType = unknownReservationType
	}
	return reservationType
}
