package task

import (
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/api"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient/models"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
)

func fillStationFromAndStationToFromPartnerResponse(segments []*TrainSegment, imRequest *models.SearchTrainPricingRequest,
	logger log.Logger, raspRepo *dict.DictRepo) []*TrainSegment {
	originStation, originErr := getExpressStation(imRequest.Origin, raspRepo)

	destinationStation, destinationErr := getExpressStation(imRequest.Destination, raspRepo)

	canFillStationsFromRequest := false
	if originErr == nil && destinationErr == nil && originStation.MajorityId != expressFakeID &&
		destinationStation.MajorityId != expressFakeID {
		canFillStationsFromRequest = true
	}

	if canFillStationsFromRequest {
		originNumber, _ := strconv.Atoi(imRequest.Origin)
		destinationNumber, _ := strconv.Atoi(imRequest.Destination)
		for i := 0; i < len(segments); i++ {
			segments[i].apiSegment.StationFrom, _ = buildStation(raspRepo, originNumber)
			segments[i].apiSegment.StationTo, _ = buildStation(raspRepo, destinationNumber)
		}
	} else {
		for i := 0; i < len(segments); i++ {
			originNumber, err := strconv.Atoi(segments[i].stationFromExpressCode)
			if err != nil {
				logger.Errorf("Не нашли станцию по экспресс коду %s", segments[i].stationFromExpressCode)
			} else {
				segments[i].apiSegment.StationFrom, _ = buildStation(raspRepo, originNumber)
			}
			destinationNumber, err := strconv.Atoi(segments[i].stationToExpressCode)
			if err != nil {
				logger.Errorf("Не нашли станцию по экспресс коду %s", segments[i].stationToExpressCode)
			} else {
				segments[i].apiSegment.StationTo, _ = buildStation(raspRepo, destinationNumber)
			}
		}
	}
	return segments
}

func fillStartAndEndStationsFromPartnerResponse(segments []*TrainSegment, raspRepo *dict.DictRepo,
	logger log.Logger) []*TrainSegment {
	for _, segment := range segments {
		startStationAlias, ok := raspRepo.GetStationExpressAlias(segment.startExpressTitleOrCode)
		if !ok {
			logger.Errorf("StationExpressAlias: не нашли станцию по названию %s", segment.startExpressTitleOrCode)
		} else {
			startStation, _ := raspRepo.GetStation(startStationAlias.StationId)
			segment.startStation = startStation
		}

		endStationAlias, ok := raspRepo.GetStationExpressAlias(segment.endExpressTitleOrCode)
		if !ok {
			logger.Errorf("StationExpressAlias: не нашли станцию по названию %s", segment.endExpressTitleOrCode)
		} else {
			endStation, _ := raspRepo.GetStation(endStationAlias.StationId)
			segment.endStation = endStation
		}
	}
	return segments
}

const providerP2 = "P2"

func buildCoachOwners(thread *rasp.TThread, raspRepo *dict.DictRepo) []string {
	if thread == nil {
		return []string{}
	}

	carrier, ok := raspRepo.GetCarrierByCarrierID(thread.CompanyId)
	if !ok {
		return []string{}
	}

	var carrierTitle string
	if carrier.ShortTitle != "" {
		carrierTitle = carrier.ShortTitle
	} else {
		carrierTitle = carrier.Title
	}

	return []string{carrierTitle}
}

func makeNoPriceSegmentsFromSearchSegment(searchSegment *RThreadSegment, raspRepo *dict.DictRepo) []*TrainSegment {
	var segments []*TrainSegment
	for key := range searchSegment.trainKeys {
		segment := TrainSegment{}
		segment.canSupplySegments = false
		segment.key = key
		segment.originalNumber = searchSegment.thread.raspThread.Number

		segment.number = searchSegment.number
		//rtstationFrom := searchSegment.rtstationFrom
		//TODO if rtstationFrom != nil && rtstationFrom.trainNumber

		//segment.tModel = searchSegment.thread.raspThread.tModel
		segment.departure = &searchSegment.departure
		segment.arrival = &searchSegment.arrival
		//segment.isDeluxe = searchSegment.thread.raspThread.isDeluxe
		segment.isSuburban = searchSegment.thread.raspThread.TransportType == rasp.TTransport_TYPE_SUBURBAN
		segment.searchSegment = searchSegment
		segment.thread = searchSegment.thread.raspThread
		segment.coachOwners = buildCoachOwners(searchSegment.thread.raspThread, raspRepo)
		segment.apiSegment = &api.Segment{}
		segment.apiSegment.Tariffs = &api.MinTariffs{}
		segment.apiSegment.Tariffs.Classes = nil
		segment.apiSegment.Tariffs.OptionalBrokenClassesCode = &api.MinTariffs_BrokenClassesCode{BrokenClassesCode: api.BrokenClassesCode_BROKEN_CLASSES_CODE_SOLD_OUT}
		segments = append(segments, &segment)
	}
	return segments
}

var trainPurchaseP2TrainNumbersMap = map[string][]string{
	"7043": {"843М", "843Х"},
	"7044": {"844Х"},
	"7078": {"878М"},
	"7062": {"862М"},
	"7009": {"809М"},
	"7022": {"822М"},
	"7021": {"821Г", "821М"},
	"7082": {"882М"},
	"7027": {"827М"},
	"7040": {"840М", "840Х"},
	"7025": {"825М"},
	"7019": {"819М"},
	"7001": {"801Г", "801М"},
	"7017": {"817М"},
	"7003": {"803И", "803Х"},
	"7002": {"802М"},
	"7005": {"805М"},
	"7004": {"804Х"},
	"7007": {"807М"},
	"7011": {"811Г"},
	"7081": {"881Г"},
	"7023": {"823М"},
	"7083": {"883Г"},
	"7015": {"815М"},
	"7085": {"885Г", "885М"},
	"7084": {"884М", "884Х"},
	"7086": {"886М"},
	"7045": {"845М"},
	"7012": {"812Г"},
	"7050": {"850М", "850Х"},
	"7051": {"851Г", "851М"},
	"7071": {"871Г"},
	"7098": {"898И"},
	"7048": {"848М"},
	"7046": {"846М"},
	"7088": {"888М"},
	"7042": {"842И", "842М"},
	"7041": {"841М", "841И", "841А"},
	"7049": {"849М"},
	"7047": {"847М"},
}

func classifyPriceSegments(segments []*TrainSegment, searchSegments []*RThreadSegment, raspRepo *dict.DictRepo) (
	[]*TrainSegment, []*TrainSegment, []*TrainSegment) {
	for ind, segment := range segments {
		var matchingSearchSegments []*RThreadSegment
		trainNumbers, ok := trainPurchaseP2TrainNumbersMap[segment.originalNumber]
		if segment.provider == providerP2 && ok {
			for _, number := range trainNumbers {
				key := makeTariffSegmentKeyExact(number, segment.departure)

				for _, searchSegment := range searchSegments {
					_, ok = searchSegment.trainKeys[key]
					if ok {
						matchingSearchSegments = append(matchingSearchSegments, searchSegment)
					}
				}
				if len(matchingSearchSegments) != 0 {
					segments[ind].key = key
					break
				}
			}
		} else {
			for _, searchSegment := range searchSegments {
				_, ok := searchSegment.trainKeys[segment.key]
				if ok {
					matchingSearchSegments = append(matchingSearchSegments, searchSegment)
				}
			}
		}
		var searchSegment *RThreadSegment
		if len(matchingSearchSegments) == 0 {
			searchSegment = nil
		} else if len(matchingSearchSegments) == 1 {
			searchSegment = matchingSearchSegments[0]
		} else {
			bestSearchSegment := matchingSearchSegments[0]
			bestIsThrough := matchingSearchSegments[0].thread.raspThread.Type == rasp.TThread_TYPE_THROUGH_TRAIN
			bestFromMajorityID := matchingSearchSegments[0].stationFrom.MajorityId
			bestToMajorityID := matchingSearchSegments[0].stationTo.MajorityId
			for i := 1; i < len(matchingSearchSegments); i++ {
				isThrough := matchingSearchSegments[i].thread.raspThread.Type == rasp.TThread_TYPE_THROUGH_TRAIN
				fromMajorityID := matchingSearchSegments[i].stationFrom.MajorityId
				toMajorityID := matchingSearchSegments[i].stationTo.MajorityId
				if isThrough != bestIsThrough {
					if !isThrough && bestIsThrough {
						bestSearchSegment = matchingSearchSegments[i]
					}
					continue
				}
				if fromMajorityID != bestFromMajorityID {
					if fromMajorityID < bestFromMajorityID {
						bestSearchSegment = matchingSearchSegments[i]
					}
					continue
				}
				if toMajorityID < bestToMajorityID {
					bestSearchSegment = matchingSearchSegments[i]
				}
			}
			searchSegment = bestSearchSegment
		}
		if searchSegment != nil {
			segment.thread = searchSegment.thread.raspThread
			segment.searchSegment = searchSegment
			searchSegment.used = true
		}
	}
	var fullSegments []*TrainSegment
	var segmentsWithoutThread []*TrainSegment
	for _, segment := range segments {
		if segment.thread != nil {
			fullSegments = append(fullSegments, segment)
		} else {
			segmentsWithoutThread = append(segmentsWithoutThread, segment)
		}
	}
	var unusedSearchSegments []*RThreadSegment
	for _, searchSegment := range searchSegments {
		if !searchSegment.used {
			unusedSearchSegments = append(unusedSearchSegments, searchSegment)
		}
	}
	var noPriceSegments []*TrainSegment
	for _, unusedSegment := range unusedSearchSegments {
		noPriceSegments = append(noPriceSegments, makeNoPriceSegmentsFromSearchSegment(unusedSegment, raspRepo)...)
	}
	return fullSegments, segmentsWithoutThread, noPriceSegments
}

func fillFullSegments(segments []*TrainSegment, raspRepo *dict.DictRepo) []*TrainSegment {
	for _, segment := range segments {
		if segment.stationFrom == nil {
			segment.stationFrom = segment.searchSegment.stationFrom
		}
		if segment.stationTo == nil {
			segment.stationTo = segment.searchSegment.stationTo
		}
		if segment.startStation == nil {
			segment.startStation = segment.searchSegment.startStation
		}
		if segment.endStation == nil {
			segment.endStation = segment.searchSegment.endStation
		}
		segment.titleCommon = segment.thread.CommonTitle
		coachOwners := buildCoachOwners(segment.thread, raspRepo)
		if len(coachOwners) != 0 {
			segment.coachOwners = coachOwners
		}
	}

	return segments
}

func buildDefaultTitleCommon(startStation *rasp.TStation, endStation *rasp.TStation) *rasp.TThreadTitle {
	titleCommon := new(rasp.TThreadTitle)
	titleCommon.TransportType = rasp.TTransport_TYPE_TRAIN
	titleCommon.Type = rasp.TThreadTitle_TYPE_DEFAULT
	titleCommon.TitleParts = []*rasp.TThreadTitlePart{}
	startTitlePart := new(rasp.TThreadTitlePart)
	startTitlePart.SettlementId = uint32(startStation.SettlementId)
	titleCommon.TitleParts = append(titleCommon.TitleParts, startTitlePart)

	endTitlePart := new(rasp.TThreadTitlePart)
	endTitlePart.SettlementId = uint32(endStation.SettlementId)
	titleCommon.TitleParts = append(titleCommon.TitleParts, endTitlePart)
	return titleCommon
}

func fillSegmentsWithoutThread(segments []*TrainSegment) []*TrainSegment {
	for _, segment := range segments {
		if segment.startStation == nil || segment.endStation == nil {
			continue
		}
		segment.titleCommon = buildDefaultTitleCommon(segment.startStation, segment.endStation)
	}
	return segments
}

func fillNoPriceSegments(segments []*TrainSegment, req *models.SearchTrainPricingRequest, raspRepo *dict.DictRepo) []*TrainSegment {
	originNumber, _ := strconv.Atoi(req.Origin)
	destinationNumber, _ := strconv.Atoi(req.Destination)
	for i := 0; i < len(segments); i++ {
		segments[i].apiSegment.StationFrom, _ = buildStation(raspRepo, originNumber)
		segments[i].apiSegment.StationTo, _ = buildStation(raspRepo, destinationNumber)

		segments[i].departure = &segments[i].searchSegment.departure
		segments[i].arrival = &segments[i].searchSegment.arrival
		segments[i].startStation = segments[i].searchSegment.startStation
		segments[i].endStation = segments[i].searchSegment.endStation
		segments[i].titleCommon = segments[i].thread.CommonTitle
	}

	return segments
}

func digitsOnly(segmentNumber string) string {
	numberAsString := ""
	for _, letter := range segmentNumber {
		if letter >= '0' && letter <= '9' {
			numberAsString += string(letter)
		} else {
			break
		}
	}
	return numberAsString
}

func isSuburbanNumber(digitalNumber string) bool {
	return len(digitalNumber) == 3 && digitalNumber[0] == '8' || len(digitalNumber) == 4 && digitalNumber[0] == '7' &&
		digitalNumber[1] == '0'
}

func keepOnlySoldOutSegments(segments []*TrainSegment, request *models.SearchTrainPricingRequest,
	raspRepo *dict.DictRepo) []*TrainSegment {
	layout := "2006-01-02T15:04:05"
	t, _ := time.Parse(layout, request.DepartureDate)
	departure, _ := strconv.Atoi(request.Origin)
	station, _ := raspRepo.GetStationByExpressCode(int32(departure))
	tz, _ := raspRepo.GetTimeZone(station.TimeZoneId)
	localTime := time.Now()
	location, _ := time.LoadLocation(tz.Code)
	departureLocalTime := localTime.In(location)
	daysFromToday := (t.Unix() - departureLocalTime.Unix()) / 60 * 60 * 24

	var soldOutSegments []*TrainSegment
	for _, segment := range segments {
		salesDepth, ok := getSalesDepth(segment)
		if ok && daysFromToday < int64(salesDepth) {
			soldOutSegments = append(soldOutSegments, segment)
		}
	}

	return soldOutSegments
}

func filterSegmentsWithoutStationFromOrStationTo(segments []*TrainSegment, request *models.SearchTrainPricingRequest,
	logger log.Logger) []*TrainSegment {
	var segmentsWithStationFromAndStationTo []*TrainSegment
	for _, segment := range segments {
		if segment.apiSegment.StationFrom == nil || segment.apiSegment.StationTo == nil {
			//logger.Errorf("Не смогли восстановить станции отправления и(или) прибытия не показываем цены для %s",
			//	segment.originalNumber)
		} else {
			segmentsWithStationFromAndStationTo = append(segmentsWithStationFromAndStationTo, segment)
		}
	}

	return segmentsWithStationFromAndStationTo
}

func fillFirstAndLastCountryCodes(segments []*TrainSegment, raspRepo *dict.DictRepo) []*TrainSegment {
	for _, segment := range segments {
		var firstCountryCode *string
		if segment.startStation != nil {
			country, ok := raspRepo.GetCountryByCountryID(int32(segment.startStation.CountryId))
			if ok {
				firstCountryCode = &country.Code
			}
		} else {
			firstCountryCode = nil
		}
		var lastCountryCode *string
		if segment.endStation != nil {
			country, ok := raspRepo.GetCountryByCountryID(int32(segment.endStation.CountryId))
			if ok {
				lastCountryCode = &country.Code
			}
		} else {
			lastCountryCode = nil
		}
		segment.firstCountryCode = firstCountryCode
		segment.lastCountryCode = lastCountryCode
		if firstCountryCode == nil && lastCountryCode == nil {
			segment.canSupplySegments = false
		}
	}
	return segments
}

func BuildPriceSegmentsWithReasonForMissingPrices(segments []*TrainSegment, searchSegments []*RThreadSegment,
	request *models.SearchTrainPricingRequest, logger log.Logger, raspRepo *dict.DictRepo) []*TrainSegment {
	segments = fillStationFromAndStationToFromPartnerResponse(segments, request, logger, raspRepo)
	segments = fillStartAndEndStationsFromPartnerResponse(segments, raspRepo, logger)
	fullSegments, segmentsWithoutThread, noPriceSegments := classifyPriceSegments(segments, searchSegments, raspRepo)
	segments = fillFullSegments(fullSegments, raspRepo)
	segments = append(segments, fillSegmentsWithoutThread(segmentsWithoutThread)...)
	noPriceSegments = fillNoPriceSegments(noPriceSegments, request, raspRepo)
	segments = append(segments, keepOnlySoldOutSegments(noPriceSegments, request, raspRepo)...)
	segments = filterSegmentsWithoutStationFromOrStationTo(segments, request, logger)
	segments = fillFirstAndLastCountryCodes(segments, raspRepo)
	return segments
}

const totalFee = 0.11

func getBeddingTariff(tariffCoachType coachType, serviceTariff float64) float64 {
	if tariffCoachType == platzkarteCoachType {
		return serviceTariff
	}
	return 0
}

type TicketCost struct {
	amount           float64
	yandexFeePercent float64
	beddingAmount    float64
	mainFee          float64
	beddingFee       float64
}

func calculateTicketCost(tariffCoachType coachType, amount float64, serviceAmount float64) *TicketCost {
	yandexFeePercent := totalFee
	var beddingAmount float64
	if amount > serviceAmount {
		beddingAmount = getBeddingTariff(tariffCoachType, serviceAmount)
	} else {
		beddingAmount = 0
	}

	return &TicketCost{
		amount:           amount,
		yandexFeePercent: yandexFeePercent,
		beddingAmount:    beddingAmount,
		mainFee:          0,
		beddingFee:       0,
	}
}

func calculateFee(currentTariff *trainTariff) {
	if currentTariff.ticketPrice == nil {
		currentTariff.fee = currentTariff.ticketPrice
		return
	}
	tariff := currentTariff.ticketPrice.value
	var serviceTariff float64
	if currentTariff.servicePrice != nil {
		serviceTariff = currentTariff.servicePrice.value
	} else {
		serviceTariff = 0
	}
	ticketCost := calculateTicketCost(currentTariff.coachType, tariff, serviceTariff)
	currentTariff.fee = &Price{
		value:    ticketCost.mainFee + ticketCost.beddingFee,
		currency: currentTariff.ticketPrice.currency,
	}
	currentTariff.feePercent = ticketCost.yandexFeePercent
}

func fillSegmentFees(segments []*TrainSegment) []*TrainSegment {
	for _, segment := range segments {
		for _, tariff := range segment.tariffsClasses {
			calculateFee(&tariff)
		}
	}
	return segments
}
