package task

import (
	"container/list"
	"errors"
	"fmt"
	"sync"

	"github.com/golang/protobuf/proto"

	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"
	tpb "a.yandex-team.ru/travel/proto"
	wpb "a.yandex-team.ru/travel/trains/worker/api"
)

var supportedPriorities = []wpb.ERequestPriority{
	wpb.ERequestPriority_REQUEST_PRIORITY_HIGH,
	wpb.ERequestPriority_REQUEST_PRIORITY_NORMAL,
	wpb.ERequestPriority_REQUEST_PRIORITY_LOW,
}
var ErrNoTasks = errors.New("no more tasks")

// NOTE: PriorityQueue is not thread safe
type PriorityQueue struct {
	queues      map[wpb.ERequestPriority]*list.List
	metricsLen  map[wpb.ERequestPriority]coreMetrics.Gauge
	metricsPush map[wpb.ERequestPriority]coreMetrics.Counter
	metricsPop  map[wpb.ERequestPriority]coreMetrics.Counter
}

func NewPriorityQueue(appMetrics *metrics.AppMetrics) *PriorityQueue {
	q := PriorityQueue{
		queues:      make(map[wpb.ERequestPriority]*list.List),
		metricsLen:  make(map[wpb.ERequestPriority]coreMetrics.Gauge),
		metricsPush: make(map[wpb.ERequestPriority]coreMetrics.Counter),
		metricsPop:  make(map[wpb.ERequestPriority]coreMetrics.Counter),
	}
	for _, priority := range supportedPriorities {
		q.queues[priority] = list.New()
		q.metricsLen[priority] = appMetrics.GetOrCreateGauge(
			"task_queue",
			map[string]string{
				"priority": priority.String(),
			},
			"len")
		q.metricsPush[priority] = appMetrics.GetOrCreateCounter(
			"task_queue",
			map[string]string{
				"priority": priority.String(),
			},
			"push_count")
		q.metricsPop[priority] = appMetrics.GetOrCreateCounter(
			"task_queue",
			map[string]string{
				"priority": priority.String(),
			},
			"pop_count")
	}
	return &q
}

// Notes
// HIGH:   sync requests
// NORMAL: user generated
// LOW:    warmer generated
func (q *PriorityQueue) Push(task proto.Message, priority wpb.ERequestPriority) (int, error) {
	queue, ok := q.queues[priority]
	if !ok {
		return -1, fmt.Errorf("PriorityQueue.Push: unknown priority=%s", priority)
	}
	queue.PushBack(task)
	queueLen := queue.Len()
	metricLen := q.metricsLen[priority]
	metricLen.Set(float64(queueLen))
	metricPush := q.metricsPush[priority]
	metricPush.Inc()
	return queueLen, nil
}

func (q *PriorityQueue) Pop() (proto.Message, error) {
	for _, priority := range supportedPriorities {
		queue, ok := q.queues[priority]
		if !ok {
			return nil, fmt.Errorf("PriorityQueue.Pop: unknown priority=%s", priority)
		}
		if queue.Len() > 0 {
			element := queue.Front()
			queue.Remove(element)
			metricLen := q.metricsLen[priority]
			metricLen.Set(float64(queue.Len()))
			metricPop := q.metricsPop[priority]
			metricPop.Inc()
			return element.Value.(proto.Message), nil
		}
	}
	return nil, ErrNoTasks
}

func (q *PriorityQueue) GetList(priority wpb.ERequestPriority) (*list.List, error) {
	queue, ok := q.queues[priority]
	if !ok {
		return nil, fmt.Errorf("PriorityQueue.GetList: unknown priority=%s", priority)
	}
	return queue, nil
}

func (q *PriorityQueue) Len() int {
	len := 0
	for _, queue := range q.queues {
		len += queue.Len()
	}
	return len
}

type SearchKey struct {
	From      string
	To        string
	DateYear  int32
	DateMonth int32
	DateDay   int32
}

func NewSearchKey(from string, to string, date *tpb.TDate) SearchKey {
	return SearchKey{
		From:      from,
		To:        to,
		DateYear:  date.Year,
		DateMonth: date.Month,
		DateDay:   date.Day,
	}
}

type SearchTaskQueue struct {
	priorityQueue *PriorityQueue
	index         map[wpb.ERequestPriority]map[SearchKey]int
	mutex         sync.Mutex
	appMetrics    *metrics.AppMetrics
}

func NewSearchTaskQueue(appMetrics *metrics.AppMetrics) *SearchTaskQueue {
	sq := SearchTaskQueue{
		priorityQueue: NewPriorityQueue(appMetrics),
		index:         make(map[wpb.ERequestPriority]map[SearchKey]int),
		appMetrics:    appMetrics,
	}

	for _, priority := range supportedPriorities {
		sq.index[priority] = make(map[SearchKey]int)
	}
	return &sq
}

func (sq *SearchTaskQueue) Push(request *wpb.TSegmentsRequest) (int, error) {
	if request.Date == nil || request.Header == nil {
		return -1, fmt.Errorf("SearchTaskQueue.Push: bad request")
	}

	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	searchKey := NewSearchKey(request.From, request.To, request.Date)
Loop:
	for _, priority := range supportedPriorities {
		priorityIndex := sq.index[priority]
		i, ok := priorityIndex[searchKey]
		if !ok {
			continue
		}
		if priority >= request.Header.Priority {
			return i, nil
		}
		delete(priorityIndex, searchKey)
		priorityQueue := sq.priorityQueue

		priorityList, err := priorityQueue.GetList(priority)
		if err != nil {
			return -1, err
		}
		for element := priorityList.Back(); element != nil; element = element.Next() {
			requestInList := element.Value.(*wpb.TSegmentsRequest)
			if searchKey == NewSearchKey(requestInList.From,
				requestInList.To, requestInList.Date) {
				priorityList.Remove(element)
				break Loop
			}
		}
		return -1, fmt.Errorf("SearchTaskQueue.Push: internal error, index and data are not synced")
	}
	priorityQueue := sq.priorityQueue

	i, err := priorityQueue.Push(request, request.Header.Priority)
	if err != nil {
		return -1, err
	}
	priorityIndex := sq.index[request.Header.Priority]
	priorityIndex[searchKey] = i
	return i, nil
}

func (sq *SearchTaskQueue) Pop() (*wpb.TSegmentsRequest, error) {
	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	priorityQueue := sq.priorityQueue

	message, err := priorityQueue.Pop()
	if err != nil {
		return nil, err
	}
	request := message.(*wpb.TSegmentsRequest)
	searchKey := NewSearchKey(request.From, request.To, request.Date)
	for _, priority := range supportedPriorities {
		priorityIndex := sq.index[priority]
		delete(priorityIndex, searchKey)
	}
	return request, nil
}

func (sq *SearchTaskQueue) Len() (int, error) {
	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	priorityQueue := sq.priorityQueue

	return priorityQueue.Len(), nil
}
