package task

import (
	"fmt"
	"sort"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/trains/search_api/api"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient/models"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
)

const stationMajorityID = 4
const trainOrderDefaultDepthOfSales = 120
const trainCode = "train"
const trainTitle = "поезд"

type StationPoint struct {
	pointCode        int
	pointExpressCode int
}

type Query struct {
	pointFrom     *StationPoint
	pointTo       *StationPoint
	departureDate time.Time
}

type TrainNumber struct {
	digits  string
	letters string
}

type RThreadSegment struct {
	stationFrom          *rasp.TStation
	stationTo            *rasp.TStation
	rtstationFrom        *rasp.TThreadStation
	rtstationTo          *rasp.TThreadStation
	rtstationFromStation *rasp.TStation
	rtstationToStation   *rasp.TStation
	rtstationFromThread  *Thread
	rtstationToThread    *Thread
	departure            time.Time
	arrival              time.Time
	thread               *Thread
	startDate            time.Time
	now                  time.Time
	trainNumber          *TrainNumber
	title                string
	subSegments          []*RThreadSegment
	trainKeys            map[string]bool
	minArrival           time.Time
	maxArrival           time.Time
	number               string
	letters              map[string]bool
	startStationID       int32
	endStationID         int32
	startStation         *rasp.TStation
	endStation           *rasp.TStation
	used                 bool
}

type TrainSegment struct {
	apiSegment              *api.Segment
	rawTrainCategory        string
	rawTrainName            string
	originalNumber          string
	number                  string
	trainNumberToGetRoute   string
	possibleNumbers         []string
	tariffs                 map[string]bool
	canSupplySegments       bool
	ufsTitle                string
	coachOwners             []string
	hasDynamicPricing       bool
	twoStorey               bool
	isSuburban              bool
	provider                string
	departure               *time.Time
	arrival                 *time.Time
	isDeluxe                bool
	stationFromExpressCode  string
	stationToExpressCode    string
	startExpressTitleOrCode string
	endExpressTitleOrCode   string
	railwayDeparture        *time.Time
	railwayArrival          *time.Time
	key                     string
	thread                  *rasp.TThread
	searchSegment           *RThreadSegment
	stationFrom             *rasp.TStation
	stationTo               *rasp.TStation
	startStation            *rasp.TStation
	endStation              *rasp.TStation
	oldUfsOrder             bool
	title                   *rasp.TThreadTitle
	titleCommon             *rasp.TThreadTitle
	firstCountryCode        *string
	lastCountryCode         *string
	tariffsClasses          map[coachType]trainTariff
	brokenClasses           map[coachType][]tariffError
}

const expressKey = 2

func getExpressCodeByStationCode(stationCode int, raspRepo *dict.DictRepo) (int, error) {
	station, ok := raspRepo.GetStation(int32(stationCode))
	if !ok {
		return 0, fmt.Errorf("can not get station by stationCode %d", stationCode)
	}
	value, ok := station.StationCodes[expressKey]
	if !ok {
		return 0, fmt.Errorf("can not get expressCode by station")
	}
	expressCode, _ := strconv.Atoi(value)
	return expressCode, nil
}

func getRailwayTzByCode(stationCode int, raspRepo *dict.DictRepo) (string, error) {
	station, ok := raspRepo.GetStationByExpressCode(int32(stationCode))
	if !ok {
		return "", fmt.Errorf("can not get station by stationCode %d", stationCode)
	}

	railwayTimeZoneID := station.RailwayTimeZoneId
	railwayTZ, ok := raspRepo.GetTimeZone(railwayTimeZoneID)

	if !ok {
		return "", fmt.Errorf("can not get timezone by railwayTimezoneID %d", railwayTimeZoneID)
	}

	return railwayTZ.Code, nil
}

func railwayTz(point StationPoint, raspRepo *dict.DictRepo) (bool, error) {
	tz, err := getRailwayTzByCode(point.pointCode, raspRepo)
	if err != nil {
		return false, err
	}
	return tz != "", nil
}

func getLocalToday(point StationPoint, raspRepo *dict.DictRepo) (time.Time, error) {
	station, ok := raspRepo.GetStationByExpressCode(int32(point.pointCode))
	if !ok {
		return time.Time{}, fmt.Errorf("can not get station by stationCode %d", point.pointCode)
	}
	tz, ok := raspRepo.GetTimeZone(station.TimeZoneId)
	if !ok {
		return time.Time{}, fmt.Errorf("can not get timezone by timezoneID %d", station.TimeZoneId)
	}

	localTime := time.Now()

	location, _ := time.LoadLocation(tz.Code)
	departureLocalTime := localTime.In(location)
	return departureLocalTime, nil
}

func canSendLocalQuery(query Query, raspRepo *dict.DictRepo) error {
	ok, err := railwayTz(*query.pointFrom, raspRepo)
	if err != nil {
		return err
	}
	if !(query.pointFrom != nil && query.pointTo != nil && ok) {
		return fmt.Errorf("incorrect point in query")
	}

	localToday, err := getLocalToday(*query.pointFrom, raspRepo)
	if err != nil {
		return err
	}

	diffInDays := (query.departureDate.Unix() - localToday.Unix()) / (24 * 60 * 60)
	if diffInDays >= 0 && diffInDays <= trainOrderDefaultDepthOfSales*24*60*60 {
		return nil
	}

	return fmt.Errorf("incorrect date in query")
}

func isThroughTrain(segment *RThreadSegment) bool {
	return segment.thread != nil && segment.thread.raspThread.Type == rasp.TThread_TYPE_THROUGH_TRAIN
}

func chooseMainSegment(segments []*RThreadSegment) *RThreadSegment {
	mainSegment := segments[0]
	mainIsThrough := isThroughTrain(segments[0])
	mainDuration := segments[0].arrival.Unix() - segments[0].departure.Unix()
	for i := 1; i < len(segments); i++ {
		isThrough := isThroughTrain(segments[i])
		duration := segments[i].arrival.Unix() - segments[i].departure.Unix()
		if !isThrough && mainIsThrough || isThrough == mainIsThrough && duration < mainDuration {
			mainSegment = segments[i]
			mainIsThrough = isThrough
			mainDuration = duration
		}
	}
	return mainSegment
}

func parseTrainNumber(number string) *TrainNumber {
	matched := trainNumberRe.MatchString(number)
	if !matched {
		return nil
	}
	groups := trainNumberRe.FindStringSubmatch(number)
	return &TrainNumber{
		digits:  groups[1],
		letters: groups[2],
	}
}

func fillTrainNumbers(segments []*RThreadSegment) {
	for _, segment := range segments {
		segment.trainNumber = parseTrainNumber(segment.thread.raspThread.Number)
	}
}

func createMetaTrain(segment *RThreadSegment, segments []*RThreadSegment) *RThreadSegment {
	metaTrain := segment
	metaTrain.subSegments = segments
	metaTrain.minArrival = segments[0].arrival
	metaTrain.maxArrival = segments[0].arrival
	for i := 1; i < len(segments); i++ {
		if segments[i].arrival.Unix() < metaTrain.minArrival.Unix() {
			metaTrain.minArrival = segments[i].arrival
		}
		if segments[i].arrival.Unix() > metaTrain.maxArrival.Unix() {
			metaTrain.maxArrival = segments[i].arrival
		}
	}

	letters := make(map[string]bool)
	for i := 0; i < len(segments); i++ {
		for _, rune_ := range segments[i].number {
			letters[string(rune_)] = true
		}
	}
	var lettersList []string

	for key := range letters {
		lettersList = append(lettersList, key)
	}
	sort.Slice(lettersList, func(i, j int) bool {
		return lettersList[i] > lettersList[j]
	})
	allLetters := strings.Join(lettersList, "")
	metaTrain.number = fmt.Sprintf("%s%s", metaTrain.trainNumber.digits, allLetters)
	metaTrain.letters = letters
	metaTrain.letters[allLetters] = true

	return metaTrain
}

func makeMetaTrains(segments []*RThreadSegment) []*RThreadSegment {
	fillTrainNumbers(segments)
	trainNumbers := map[TrainNumber][]*RThreadSegment{}
	for i := 0; i < len(segments); i++ {
		if segments[i].trainNumber != nil {
			trainNumbers[*segments[i].trainNumber] = append(trainNumbers[*segments[i].trainNumber], segments[i])
		}
	}
	var answerSegments []*RThreadSegment
	for _, value := range trainNumbers {
		if len(value) >= 2 {
			mainSegment := chooseMainSegment(value)

			metaTrain := createMetaTrain(mainSegment, value)
			answerSegments = append(answerSegments, metaTrain)
		} else {
			answerSegments = append(answerSegments, value[0])
		}
	}
	return answerSegments
}

func getPossibleNumbers(number string) []string {
	matched := trainNumberRe.MatchString(number)
	if !matched {
		return []string{number}
	}
	stringNumber, letter := getNumberAndFirstLetter(number)
	intNumber, _ := strconv.Atoi(stringNumber)
	reverseIntNumber, _ := strconv.Atoi(getReverseTrainNumber(stringNumber))
	return []string{fmt.Sprintf("%03d%s", intNumber, letter),
		fmt.Sprintf("%03d%s", reverseIntNumber, letter)}
}

func getReverseTrainNumber(digits string) string {
	intDigits, _ := strconv.Atoi(digits)
	if intDigits%2 == 0 {
		return strconv.Itoa(intDigits - 1)
	}
	return strconv.Itoa(intDigits + 1)
}

const timeRangeHours = 2

func makeDatetimeKey(dt time.Time) string {
	alignedDt := time.Date(dt.Year(), dt.Month(), dt.Day(), dt.Hour()/timeRangeHours*timeRangeHours, dt.Minute(),
		dt.Second(), dt.Nanosecond(), dt.Location())
	return fmt.Sprintf("%d%d%d_%d", alignedDt.Year(), alignedDt.Month(), alignedDt.Day(), alignedDt.Hour())
}

func makePreciseDatetimeKey(dt time.Time) string {
	return fmt.Sprintf("%d%d%d_%d%d", dt.Year(), dt.Month(), dt.Day(), dt.Hour(), dt.Minute())
}

func makeDatetimeKeys(segment *RThreadSegment, numbers []string) []string {
	var dtKeys []string

	dtKeys = append(dtKeys, makeDatetimeKey(segment.departure.Add(time.Hour*timeRangeHours/2)))
	dtKeys = append(dtKeys, makeDatetimeKey(segment.departure.Add(-(time.Hour * timeRangeHours / 2))))

	sort.Slice(dtKeys, func(i, j int) bool {
		return dtKeys[i] < dtKeys[j]
	})
	var keys []string
	for _, number := range numbers {
		for _, key := range dtKeys {
			keys = append(keys, fmt.Sprintf("train %s %s", number, key))
		}
	}
	return keys
}

func makeSegmentTrainKeys(segment *RThreadSegment) []string {
	numbers := getPossibleNumbers(segment.number)

	var digits []string
	if segment.letters != nil && segment.trainNumber != nil {
		digits = append(digits, segment.trainNumber.digits)
		digits = append(digits, getReverseTrainNumber(segment.trainNumber.digits))
		numbers = []string{}
		for letter := range segment.letters {
			for j := 0; j < len(digits); j++ {
				numbers = append(numbers, fmt.Sprintf("%s%s", digits[j], letter))
			}
		}
	}
	return makeDatetimeKeys(segment, numbers)
}

func getSearchSegmentsWithKeys(request *models.SearchTrainPricingRequest, raspRepo *dict.DictRepo) []*RThreadSegment {
	fromExpressCode, _ := strconv.Atoi(request.Origin)
	toExpressCode, _ := strconv.Atoi(request.Destination)
	searchSegments, err := searchRoutes(fromExpressCode, toExpressCode, request.DepartureDate, raspRepo)
	if err != nil {
		return nil
	}
	searchSegments = makeMetaTrains(searchSegments)
	for _, segment := range searchSegments {
		trainKeys := makeSegmentTrainKeys(segment)
		setTrainKeys := make(map[string]bool)
		for i := 0; i < len(trainKeys); i++ {
			setTrainKeys[trainKeys[i]] = true
		}
		segment.trainKeys = setTrainKeys
	}
	return searchSegments
}

type Thread struct {
	raspThread    *rasp.TThread
	stationFrom   *rasp.TStation
	stationTo     *rasp.TStation
	rtstationFrom *rasp.TThreadStation
	rtstationTo   *rasp.TThreadStation
}

func getBestThread(threads []*Thread) *Thread {
	bestThread := threads[0]
	for i := 1; i < len(threads); i++ {
		if threads[i].stationFrom.MajorityId < bestThread.stationFrom.MajorityId {
			bestThread = threads[i]
			continue
		} else if threads[i].stationFrom.MajorityId > bestThread.stationFrom.MajorityId {
			continue
		}
		if threads[i].stationTo.MajorityId < bestThread.stationTo.MajorityId {
			bestThread = threads[i]
			continue
		} else if threads[i].stationTo.MajorityId > bestThread.stationTo.MajorityId {
			continue
		}

		if threads[i].rtstationFrom.DepartureTz > bestThread.rtstationFrom.DepartureTz {
			bestThread = threads[i]
			continue
		} else if threads[i].rtstationFrom.DepartureTz < bestThread.rtstationFrom.DepartureTz {
			continue
		}

		if threads[i].rtstationTo.ArrivalTz < bestThread.rtstationTo.ArrivalTz {
			bestThread = threads[i]
		}
	}

	return bestThread
}

func removeDuplicates(threads []*Thread) []*Thread {
	sort.Slice(threads, func(i, j int) bool {
		return threads[i].raspThread.Id < threads[j].raspThread.Id
	})
	var uniqueThreads []*Thread
	var currentGroup []*Thread
	for i := 0; i < len(threads); i++ {
		if i == 0 || threads[i].raspThread.Id == threads[i-1].raspThread.Id {
			currentGroup = append(currentGroup, threads[i])
		} else {
			uniqueThreads = append(uniqueThreads, getBestThread(currentGroup))
			currentGroup = nil
			currentGroup = append(currentGroup, threads[i])
		}
	}

	if len(currentGroup) != 0 {
		uniqueThreads = append(uniqueThreads, getBestThread(currentGroup))
	}

	return uniqueThreads
}

func getThreads(preparedThreads []*Thread) map[int][]*Thread {
	var threads []*Thread
	for i := 0; i < len(preparedThreads); i++ {
		if preparedThreads[i].raspThread.Type != rasp.TThread_TYPE_INTERVAL && preparedThreads[i].raspThread.Type != rasp.TThread_TYPE_CANCEL {
			threads = append(threads, preparedThreads[i])
		}
	}

	threads = removeDuplicates(threads)
	byRtsTimeZone := map[int][]*Thread{}

	for i := 0; i < len(threads); i++ {
		byRtsTimeZone[int(threads[i].rtstationFrom.TimeZoneId)] = append(byRtsTimeZone[int(threads[i].rtstationFrom.TimeZoneId)], threads[i])
	}
	return byRtsTimeZone
}

const minutesInDay = 24 * 60

type PreSegment struct {
	thread                   Thread
	rtsFromDepartureDuration time.Duration
	rtsTzTo                  int
	rtsMaskShift             int32
	pseudoDurationTd         time.Duration
}

func getSingleZonePresegments(zoneThreads []*Thread) []*PreSegment {
	var presegments []*PreSegment
	for i := 0; i < len(zoneThreads); i++ {
		rtsFrom := zoneThreads[i].rtstationFrom
		rtsTo := zoneThreads[i].rtstationTo

		thread := zoneThreads[i]
		rtsFromDepartureDuration := time.Duration(int(zoneThreads[i].raspThread.TzStartTime/60+int32(rtsFrom.DepartureTz)) * int(time.Minute))
		rtsTzTo := rtsTo.TimeZoneId
		maskShift := (zoneThreads[i].raspThread.TzStartTime/60 + int32(rtsFrom.DepartureTz)) / minutesInDay
		pseudoDurationTd := time.Duration(int(rtsTo.ArrivalTz-rtsFrom.DepartureTz) * int(time.Minute))
		presegments = append(presegments, &PreSegment{
			thread:                   *thread,
			rtsFromDepartureDuration: rtsFromDepartureDuration,
			rtsTzTo:                  int(rtsTzTo),
			rtsMaskShift:             maskShift,
			pseudoDurationTd:         pseudoDurationTd,
		})
	}
	presegments = removeThroughTrains(presegments)
	return presegments
}

func isEqualPreSegment(firstSegment *PreSegment, secondSegment *PreSegment) bool {
	return firstSegment.thread.raspThread.TransportType == secondSegment.thread.raspThread.TransportType &&
		firstSegment.thread.stationFrom.Id == secondSegment.thread.stationFrom.Id &&
		firstSegment.rtsFromDepartureDuration == secondSegment.rtsFromDepartureDuration &&
		firstSegment.thread.stationTo.Id == secondSegment.thread.stationTo.Id &&
		firstSegment.pseudoDurationTd == secondSegment.pseudoDurationTd
}

func removeThroughTrains(results []*PreSegment) []*PreSegment {

	sort.Slice(results, func(i, j int) bool {
		if results[i].thread.raspThread.TransportType != results[j].thread.raspThread.TransportType {
			return results[i].thread.raspThread.TransportType < results[j].thread.raspThread.TransportType
		}
		if results[i].thread.stationFrom.Id != results[j].thread.stationFrom.Id {
			return results[i].thread.stationFrom.Id < results[j].thread.stationFrom.Id
		}
		if results[i].rtsFromDepartureDuration != results[j].rtsFromDepartureDuration {
			return results[i].rtsFromDepartureDuration < results[j].rtsFromDepartureDuration
		}
		if results[i].thread.stationTo.Id != results[j].thread.stationTo.Id {
			return results[i].thread.stationTo.Id < results[j].thread.stationTo.Id
		}
		return results[i].pseudoDurationTd < results[j].pseudoDurationTd
	})

	var removedResults []*PreSegment
	var currentGroup []*PreSegment
	for i := 0; i < len(results); i++ {
		currentGroup = append(currentGroup, results[i])
		if i == len(results)-1 || !isEqualPreSegment(results[i], results[i+1]) {
			if len(currentGroup) == 1 {
				removedResults = append(removedResults, currentGroup[0])
				currentGroup = nil
				continue
			}
			var basicPreSegments []*PreSegment
			for j := 0; j < len(currentGroup); j++ {
				if currentGroup[j].thread.raspThread.Type == rasp.TThread_TYPE_BASIC {
					basicPreSegments = append(basicPreSegments, currentGroup[j])
				}
			}

			if len(basicPreSegments) > 0 {
				currentGroup = basicPreSegments
			}

			removedResults = append(removedResults, currentGroup...)
			currentGroup = nil
		}
	}

	return removedResults
}

type PreSegmentAware struct {
	thread          Thread
	threadStartDate time.Time
	locDepartureDt  time.Time
	locArrivalDt    time.Time
	duration        time.Duration
}

func runsAt(thread *rasp.TThread, date time.Time) bool {
	if len(thread.YearDays) <= int(date.Month()-1) {
		return false
	}
	days := thread.YearDays[date.Month()-1]
	day := date.Day()
	return ((days >> (day - 1)) % 2) == 1
}

func getSingleZoneIter(zoneThreads []*Thread, zone int, fromDT time.Time, raspRepo *dict.DictRepo) []*PreSegmentAware {
	preSegments := getSingleZonePresegments(zoneThreads)
	tzcode, _ := strconv.Atoi(zoneThreads[0].stationFrom.TimeZoneCode)
	tz, _ := raspRepo.GetTimeZone(int32(tzcode))
	outTzFrom, _ := time.LoadLocation(tz.Code)

	tzcode, _ = strconv.Atoi(zoneThreads[0].stationTo.TimeZoneCode)
	tz, _ = raspRepo.GetTimeZone(int32(tzcode))
	outTzTo, _ := time.LoadLocation(tz.Code)

	rtsTz, _ := raspRepo.GetTimeZone(int32(zone))
	location, _ := time.LoadLocation(rtsTz.Code)
	startDt := fromDT.In(location)
	startDt = time.Date(startDt.Year(), startDt.Month(), startDt.Day(), 0, 0, 0, 0, location)

	boundDt := startDt.AddDate(0, 0, -10)

	limit := startDt.AddDate(0, 0, 100)
	var results []*PreSegmentAware
	for {
		for i := 0; i < len(preSegments); i++ {
			locDt := boundDt.Add(preSegments[i].rtsFromDepartureDuration)
			if locDt.Unix() < startDt.Unix() {
				continue
			}

			threadStartDate := startDt
			var presegmAwr PreSegmentAware
			presegmAwr.thread = preSegments[i].thread
			presegmAwr.threadStartDate = threadStartDate
			presegmAwr.locDepartureDt = locDt.In(outTzFrom)

			presegmAwr.locArrivalDt = locDt.Add(preSegments[i].pseudoDurationTd).In(outTzTo)
			presegmAwr.duration = time.Duration(presegmAwr.locDepartureDt.Unix() - presegmAwr.locArrivalDt.Unix())
			if !runsAt(preSegments[i].thread.raspThread, boundDt) {
				continue
			}
			results = append(results, &presegmAwr)
		}

		boundDt = boundDt.Add(time.Hour * 24)
		if boundDt.Unix() > limit.Unix() {
			break
		}
	}

	return results
}

func genPreSegmentsFromDt(fromDT time.Time, preparedThreads []*Thread, raspRepo *dict.DictRepo) []*PreSegmentAware {
	threadsByRtsTimeZone := getThreads(preparedThreads)
	var threadsArray []*PreSegmentAware
	for zone, threadsInZone := range threadsByRtsTimeZone {
		threadsArray = append(threadsArray, getSingleZoneIter(threadsInZone, zone, fromDT, raspRepo)...)
	}

	sort.Slice(threadsArray, func(i, j int) bool {
		return threadsArray[i].locDepartureDt.Unix() < threadsArray[j].locDepartureDt.Unix()
	})

	return threadsArray
}

func initData(segment *RThreadSegment) {
	//TODO
	if segment.thread != nil {
		segment.number = segment.thread.raspThread.Number
		//segment.title = segment.thread.raspThread.CommonTitle.String()
	}
}

func search(fromDT time.Time, toDT time.Time, preparedThreads []*Thread,
	raspRepo *dict.DictRepo) ([]*RThreadSegment, error) {
	var segments []*RThreadSegment
	now := time.Now()
	mskTZ, _ := time.LoadLocation("Europe/Moscow")
	mskNow := now.In(mskTZ)
	preSegmentsFromDt := genPreSegmentsFromDt(fromDT, preparedThreads, raspRepo)
	for _, preSegment := range preSegmentsFromDt {
		segment := RThreadSegment{}
		thread := preSegment.thread

		segment.stationFrom = thread.stationFrom
		segment.stationTo = thread.stationTo

		segment.thread = &thread

		segment.departure = preSegment.locDepartureDt
		segment.arrival = preSegment.locArrivalDt

		segment.startDate = preSegment.threadStartDate

		segment.rtstationFrom = thread.rtstationFrom
		segment.rtstationFromStation = segment.stationFrom
		segment.rtstationFromThread = &thread

		segment.rtstationTo = thread.rtstationTo
		segment.rtstationToStation = segment.stationTo
		segment.rtstationToThread = &thread

		//segment.stops_translations = thread.stops_translations

		segment.now = mskNow

		initData(&segment)
		if segment.departure.Unix() > toDT.Unix() {
			break
		}
		segments = append(segments, &segment)
	}
	return segments, nil
}

func searchRoutes(fromExpressCode int, toExpressCode int, date string, raspRepo *dict.DictRepo) ([]*RThreadSegment, error) {
	const funcName = "searchRoutes"
	preparedThreads := getThreadsFromZnoderoute(fromExpressCode, toExpressCode, raspRepo)
	fromDT, toDT, err := getLocSearchRangeAware(fromExpressCode, date, raspRepo)
	if err != nil {
		return nil, fmt.Errorf("%s: can not get local range of the departure time", funcName)
	}

	segments, err := search(fromDT, toDT, preparedThreads, raspRepo)
	if err != nil {
		return nil, err
	}
	return segments, nil
}

func getLocSearchRangeAware(fromExpressCode int, date string, raspRepo *dict.DictRepo) (time.Time, time.Time, error) {
	rangeLength := time.Hour * 24
	layout := "2006-01-02"
	locStart, _ := time.Parse(layout, date)
	locEnd := locStart.Add(rangeLength)

	station, ok := raspRepo.GetStationByExpressCode(int32(fromExpressCode))
	if !ok {
		return time.Time{}, time.Time{}, fmt.Errorf("can not get station by expressCode %d", fromExpressCode)
	}
	tz, ok := raspRepo.GetTimeZone(station.TimeZoneId)
	if !ok {
		return time.Time{}, time.Time{}, fmt.Errorf("can not get timezone by timezoneID %d", station.TimeZoneId)
	}

	location, _ := time.LoadLocation(tz.Code)
	departureLocalStart := locStart.In(location)
	departureLocalEnd := locEnd.In(location)

	return departureLocalStart, departureLocalEnd, nil
}

func getThreadsFromZnoderoute(fromExpressCode int, toExpressCode int, raspRepo *dict.DictRepo) []*Thread {
	maxMajorityID, err := getLimitConditions(fromExpressCode, toExpressCode, raspRepo)
	if err != nil {
		return nil
	}

	pointFrom, _ := raspRepo.GetStationByExpressCode(int32(fromExpressCode))
	pointTo, _ := raspRepo.GetStationByExpressCode(int32(toExpressCode))
	if pointFrom.MajorityId > int32(maxMajorityID) || pointTo.MajorityId > int32(maxMajorityID) {
		return nil
	}
	commonThreads := raspRepo.GetThreadsNumbersByStationIDs(pointFrom.Id, pointTo.Id)

	var threadsQs []*Thread
	for i := 0; i < len(commonThreads); i++ {
		stations, ok := raspRepo.GetThreadStationsByThreadID(commonThreads[i])
		if !ok {
			continue
		}
		hasPointFrom := false
		isPointToAfterPointFrom := false
		for _, station := range stations {
			if station.StationId == pointFrom.Id {
				hasPointFrom = true
			}
			if station.StationId == pointTo.Id && hasPointFrom {
				isPointToAfterPointFrom = true
			}
		}
		if !isPointToAfterPointFrom {
			continue
		}

		thread := Thread{}
		thread.raspThread, ok = raspRepo.GetThread(commonThreads[i])
		if !ok {
			continue
		}
		thread.stationFrom = pointFrom
		thread.stationTo = pointTo

		for _, station := range stations {
			if station.StationId == pointFrom.Id {
				thread.rtstationFrom = station
			}
			if station.StationId == pointTo.Id {
				thread.rtstationTo = station
			}
		}
		if thread.rtstationFrom == nil || thread.rtstationTo == nil {
			continue
		}
		threadsQs = append(threadsQs, &thread)
	}
	return threadsQs
}

func getLimitConditions(from int, to int, raspRepo *dict.DictRepo) (int, error) {
	pointFrom, ok := raspRepo.GetStationByExpressCode(int32(from))
	if !ok {
		return 0, fmt.Errorf("incorrect station %d", from)
	}
	pointTo, ok := raspRepo.GetStationByExpressCode(int32(to))
	if !ok {
		return 0, fmt.Errorf("incorrect station %d", from)
	}

	if pointFrom.SettlementId == pointTo.SettlementId {
		return 0, fmt.Errorf("from.settlementId equal to.settlementId")
	}

	return stationMajorityID, nil
}

func fillStartAndEndStationsFromThread(searchSegments []*RThreadSegment, raspRepo *dict.DictRepo) []*RThreadSegment {
	segmentsByThreadID := make(map[int32][]*RThreadSegment)
	for _, segment := range searchSegments {
		segmentsByThreadID[segment.thread.raspThread.Id] = append(segmentsByThreadID[segment.thread.raspThread.Id], segment)
	}

	for threadID, threadSegments := range segmentsByThreadID {
		stations, _ := raspRepo.GetThreadStationsByThreadID(threadID)
		stationIds := make([]int32, 0)
		for _, station := range stations {
			stationIds = append(stationIds, station.StationId)
		}

		for _, segment := range threadSegments {
			segment.startStationID = stationIds[0]
			segment.endStationID = stationIds[len(stationIds)-1]
		}
	}

	for _, segment := range searchSegments {
		segment.startStation, _ = raspRepo.GetStation(segment.startStationID)
		segment.endStation, _ = raspRepo.GetStation(segment.endStationID)
	}

	return searchSegments
}

func BeforeImQuery(request *models.SearchTrainPricingRequest, raspRepo *dict.DictRepo) []*RThreadSegment {
	searchSegments := getSearchSegmentsWithKeys(request, raspRepo)
	searchSegments = fillStartAndEndStationsFromThread(searchSegments, raspRepo)

	return searchSegments
}
