package task

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/vault"
	tariffsModels "a.yandex-team.ru/travel/trains/library/go/tariffs/models"
	apipb "a.yandex-team.ru/travel/trains/worker/api"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/clients/imclient/models"
	"a.yandex-team.ru/travel/trains/worker/internal/pkg/dict"
)

type TrainTariffsTaskConfig struct {
	ImclientBaseURL     string
	ImclientPath        string
	ImclientSecret      string
	ImclientLoginKey    string
	ImclientPasswordKey string
	ImclientPosKey      string
	Timeout             time.Duration
}

type TrainTariffsTask struct {
	ctx            context.Context
	logger         *zap.Logger
	queue          *SearchTaskQueue
	imClient       *imclient.IMClient
	searchProducer *logbroker.Producer
	metrics        *metrics.AppMetrics
	raspRepo       *dict.DictRepo
}

func NewTrainTariffsTask(
	ctx context.Context,
	queue *SearchTaskQueue,
	logger *zap.Logger,
	trainTariffsTaskConfig *TrainTariffsTaskConfig,
	searchProducer *logbroker.Producer,
	metrics *metrics.AppMetrics,
	raspRepo *dict.DictRepo,
) (*TrainTariffsTask, error) {

	r := vault.NewYavSecretsResolver()
	secLogin, err := r.GetSecretValue(trainTariffsTaskConfig.ImclientSecret, trainTariffsTaskConfig.ImclientLoginKey)
	if err != nil {
		return nil, fmt.Errorf("can not get login: %s", err)
	}
	secPassword, err := r.GetSecretValue(trainTariffsTaskConfig.ImclientSecret, trainTariffsTaskConfig.ImclientPasswordKey)
	if err != nil {
		return nil, fmt.Errorf("can not get password: %s", err)
	}
	secPos, err := r.GetSecretValue(trainTariffsTaskConfig.ImclientSecret, trainTariffsTaskConfig.ImclientPosKey)
	if err != nil {
		return nil, fmt.Errorf("can not get pos: %s", err)
	}

	imClient, _ := imclient.NewIMClient(trainTariffsTaskConfig.ImclientBaseURL, trainTariffsTaskConfig.ImclientPath,
		trainTariffsTaskConfig.Timeout, logger, secLogin, secPassword, secPos)

	return &TrainTariffsTask{
		ctx:            ctx,
		logger:         logger,
		queue:          queue,
		imClient:       imClient,
		searchProducer: searchProducer,
		metrics:        metrics,
		raspRepo:       raspRepo,
	}, nil
}

func (t *TrainTariffsTask) GetRaspRepo() *dict.DictRepo {
	return t.raspRepo
}

func (t *TrainTariffsTask) GetIMClient() *imclient.IMClient {
	return t.imClient
}

type Config struct {
	RPS         float64
	Concurrency uint32
}

var Cfg = Config{
	RPS:         1,
	Concurrency: 10,
}

func (t *TrainTariffsTask) MaxRPS() float64 {
	return Cfg.RPS
}

func (t *TrainTariffsTask) MaxConcurrency() uint32 {
	return Cfg.Concurrency
}

func (t *TrainTariffsTask) ConstructSearchTrainPricingRequest(searchRequest *apipb.TSegmentsRequest) (
	*models.SearchTrainPricingRequest, error) {
	const funcName = "constructSearchTrainPricingRequest"
	date := searchRequest.Date

	pointFromCode, err := strconv.Atoi(searchRequest.From)
	if err != nil {
		return nil, fmt.Errorf("%s. Incorrect pointFrom code", funcName)
	}

	pointToCode, err := strconv.Atoi(searchRequest.To)
	if err != nil {
		return nil, fmt.Errorf("%s: incorrect pointTo code", funcName)
	}
	query := Query{
		pointFrom: &StationPoint{
			pointCode: pointFromCode,
		},
		pointTo: &StationPoint{
			pointCode: pointToCode,
		},
		departureDate: time.Date(int(searchRequest.Date.Year), time.Month(int(searchRequest.Date.Month)),
			int(searchRequest.Date.Day), 0, 0, 0, 0, time.Local),
	}
	query.pointFrom.pointExpressCode = pointFromCode
	query.pointTo.pointExpressCode = pointToCode

	err = canSendLocalQuery(query, t.raspRepo)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", funcName, err.Error())
	} else {
		t.logger.Info("correct query")
	}

	request := &models.SearchTrainPricingRequest{
		Origin:         strconv.Itoa(pointFromCode),
		Destination:    strconv.Itoa(pointToCode),
		DepartureDate:  fmt.Sprintf("%d-%02d-%02d", date.Year, date.Month, date.Day),
		TimeFrom:       0,
		TimeTo:         24,
		GetByLocalTime: true,
	}
	return request, nil
}

func (t *TrainTariffsTask) BuildSegmentsFromIMResponse(request *models.SearchTrainPricingRequest,
	searchRequest *apipb.TSegmentsRequest) ([]*tariffsModels.DirectionTariffTrain, error) {
	const funcName = "BuildSegmentsFromIMResponse"
	t.logger.Info("BeforeImQuery")
	searchSegments := BeforeImQuery(request, t.raspRepo)
	t.logger.Info("make search segments")

	response, err := t.imClient.SearchTrainPricing(t.ctx, request.Origin, request.Destination,
		request.DepartureDate, 0, 24, true)
	t.logger.Info("get answer from IM")
	var segments []*TrainSegment
	if err == nil {
		segments = BuildTrainSegments(response, request, t.raspRepo, t.logger)
		segments = BuildPriceSegmentsWithReasonForMissingPrices(segments, searchSegments, request,
			t.logger, t.raspRepo)
	}
	var directionTariffTrains []*tariffsModels.DirectionTariffTrain
	if segments == nil {
		return nil, fmt.Errorf("%s: Segments not found", funcName)
	} else {
		t.logger.Infof("Found %d segments", len(segments))
		pointFromCode, _ := strconv.Atoi(searchRequest.From)
		pointToCode, _ := strconv.Atoi(searchRequest.To)
		directionTariffTrains = BuildSegmentsWithTariff(segments, pointFromCode, pointToCode)
		return directionTariffTrains, nil
	}
}

func (t *TrainTariffsTask) writeMessageInLogbroker(directionTariffTrains []*tariffsModels.DirectionTariffTrain) {
	const funcName = "writeMessageInLogbroker"
	message := new(apipb.TSegmentsResult)
	var err error
	if message.TariffTrain, err = tariffsModels.MapDirectionTariffTrainJSONToAPI(t.ctx,
		directionTariffTrains...); err != nil {
		t.logger.Errorf("%s: mapping json tariff failed: %v", funcName, err)
	}

	if err = t.searchProducer.Write(message); err != nil {
		t.logger.Errorf("%s: tariff writing failed: %v", funcName, err)
	}
}

func (t *TrainTariffsTask) Do() {
	const funcName = "TrainTariffs.Do"

	searchRequest, err := t.queue.Pop()
	if err != nil {
		if err != ErrNoTasks {
			t.logger.Errorf("%s: %s", funcName, err.Error())
		}
		return
	}

	request, err := t.ConstructSearchTrainPricingRequest(searchRequest)
	if err != nil {
		t.logger.Errorf(err.Error())
		return
	}

	directionTariffTrains, err := t.BuildSegmentsFromIMResponse(request, searchRequest)
	if err != nil {
		t.logger.Errorf("%s: %s", funcName, err.Error())
		return
	}

	t.writeMessageInLogbroker(directionTariffTrains)
}
