package ru.yandex.atom.db.cassandra

import com.datastax.driver.{core => cd}
import cd.{querybuilder => cqb, ConsistencyLevel, SimpleStatement}
import ru.yandex.atom.db.cassandra.querybuilder.internals.{IAssignment, WhereClause}
import ru.yandex.atom.db.cassandra.querybuilder.internals.select.CQLOrdering

/**
 * @author avhaliullin
 */
package object querybuilder {

  def statement(query: String, values: CQLValue*) = new SimpleStatement(query, values.map(_.value): _*)

  trait BuildableToStatement[T <: Statement] {
    def build: T = build(ConsistencyLevel.ONE)

    def build(consistencyLevel: ConsistencyLevel): T
  }

  object internals {

    object select {

      trait ISelect {
        private[select] def _buildSelectClause: cqb.Select.Builder

        def FROM(table: Symbol) = new FromTable(this, table.name)
      }

      object SelectCountAll extends ISelect {
        private[select] def _buildSelectClause = cqb.QueryBuilder.select().countAll()
      }

      class Select private[select](_distinct: Boolean, columns: Seq[String]) extends ISelect {
        def this(columns: Seq[String]) = this(false, columns)

        def distinct: Select = new Select(_distinct = true, columns = columns)

        private[select] def _buildSelectClause = cqb.QueryBuilder.select(columns: _*)

      }

      trait IReadySelectBuilder extends BuildableToStatement[SelectStatement] {
        private[select] def _build(cl: ConsistencyLevel): cqb.Select

        def build(cl: ConsistencyLevel) = new SelectStatement(_build(cl))
      }

      class FromTable private[select](select: ISelect, table: String) extends IReadySelectBuilder {

        def WHERE(clause: WhereClause): Where = new Where(this, _allowFiltering = false, Seq(clause))

        private[select] def _build(cl: ConsistencyLevel): cqb.Select = {
          val res = select._buildSelectClause.from(table)
          res.setConsistencyLevel(cl)
          res
        }
      }

      class Where private[select](head: IReadySelectBuilder, _allowFiltering: Boolean, clauses: Seq[WhereClause]) extends IReadySelectBuilder {
        def AND(clause: WhereClause) = new Where(head = head, _allowFiltering = _allowFiltering, clauses = clauses :+ clause)

        def ALLOW_FILTERING = new Where(head = head, clauses = clauses, _allowFiltering = true)

        def ORDER_BY(orderings: CQLOrdering*) = new OrderBy(this, orderings)

        private[select] def _build(cl: ConsistencyLevel) = {
          val start = head._build(cl)
          clauses.foldLeft(start.where())(_ and _.clause)
          //Нативные стейтменты мутабельны, поэтому оно так работает
          //А нужно это потому, что Where не является Select'ом
          start
        }
      }

      class OrderBy private[select](head: IReadySelectBuilder, ordering: Seq[CQLOrdering]) extends IReadySelectBuilder {
        def LIMIT(limit: Int) = new Limit(this, limit)

        private[select] def _build(cl: ConsistencyLevel) = head._build(cl).orderBy(ordering.map(_.ordering): _*)

      }

      class Limit(head: IReadySelectBuilder, limit: Int) extends BuildableToStatement[SelectStatement] {
        def build(cl: ConsistencyLevel) = new SelectStatement(head._build(cl).limit(limit))
      }


      case class CQLOrdering(ordering: cqb.Ordering)

    }

    object update {

      trait IUpdate {
        private[update] def _buildUpdateTable: cqb.Update

        def SET(values: IAssignment*) = new UpdateValues(this, values)
      }

      class UpdateTable(_table: String) extends IUpdate {
        override private[update] def _buildUpdateTable = cqb.QueryBuilder.update(_table)

        def USING(using: UsingClause*) = new UpdateUsing(this, using)
      }

      class UpdateUsing(head: IUpdate, using: Seq[UsingClause]) extends IUpdate {
        override private[update] def _buildUpdateTable = {
          val builder = head._buildUpdateTable
          using.foreach(clause => builder using clause.using)
          builder
        }
      }

      trait IReadyUpdate extends BuildableToStatement[UpdateStatement] {
        private[update] def _build(cl: ConsistencyLevel): cqb.Update

        def build(cl: ConsistencyLevel) = new UpdateStatement(_build(cl))
      }

      class UpdateValues(head: IUpdate, values: Seq[IAssignment]) extends IReadyUpdate {
        def WHERE(clause: WhereClause) = new UpdateWhere(this, Seq(clause))

        override private[update] def _build(cl: ConsistencyLevel) = {
          val builder = head._buildUpdateTable
          values.foreach {
            assignment =>
              builder.`with`(assignment.assignment)
          }
          builder
        }
      }

      class UpdateWhere private[update](head: IReadyUpdate, clauses: Seq[WhereClause]) extends IReadyUpdate {
        def AND(clause: WhereClause) = new UpdateWhere(head = head, clauses = clauses :+ clause)

        def IF(clause: WhereClause) = new UpdateIf(this, Seq(clause))

        private[update] def _build(cl: ConsistencyLevel) = {
          val builder = head._build(cl)
          clauses.foldLeft(builder.where())(_ and _.clause)
          builder
        }
      }

      class UpdateIf private[update](head: IReadyUpdate, clauses: Seq[WhereClause]) extends IReadyUpdate {
        def AND(clause: WhereClause) = new UpdateIf(head = head, clauses = clauses :+ clause)

        override private[update] def _build(cl: ConsistencyLevel) = {
          val builder = head._build(cl)
          clauses.foldLeft(builder.onlyIf())(_ and _.clause)
          builder
        }
      }

    }

    object insert {

      class InsertInto(_table: String) {
        def VALUES(values: (Symbol, CQLValue)*) = new InsertValues(this, values)

        private[insert] def table = _table
      }

      trait IReadyInsert extends BuildableToStatement[InsertStatement] {
        private[insert] def _build(cl: ConsistencyLevel): cqb.Insert

        def build(cl: ConsistencyLevel) = new InsertStatement(_build(cl))
      }

      class InsertValues private[insert](insertInto: InsertInto, values: Seq[(Symbol, CQLValue)]) extends IReadyInsert {
        def USING(using: UsingClause*) = new InsertUsing(this, using)

        private[insert] def _build(cl: ConsistencyLevel) = {
          val start = cqb.QueryBuilder.insertInto(insertInto.table)
          start.setConsistencyLevel(cl)
          values.foldLeft(start) {
            (builder, nameValue) =>
              builder.value(nameValue._1.name, nameValue._2.value)
          }
          start
        }
      }

      class InsertUsing private[insert](readyInsert: IReadyInsert, using: Seq[UsingClause]) extends IReadyInsert {

        private[insert] def _build(cl: ConsistencyLevel) = {
          val builder = readyInsert._build(cl)
          using.foreach(clause => builder using clause.using)
          builder
        }
      }

    }

    object delete {

      trait IDelete {
        private[delete] def _buildDeleteClause: cqb.Delete.Builder

        def FROM(table: Symbol) = new DeleteFrom(this, table.name)
      }

      class Delete(columns: String*) extends IDelete {
        private[delete] def _buildDeleteClause = cqb.QueryBuilder.delete(columns: _*)
      }

      class DeleteAll extends IDelete {
        private[delete] def _buildDeleteClause = cqb.QueryBuilder.delete()
      }

      trait IDeleteFrom {
        private[delete] def _buildFrom(cl: ConsistencyLevel): cqb.Delete

        def WHERE(clause: WhereClause) = DeleteWhere(this, Seq(clause))
      }

      class DeleteFrom private[delete](delete: IDelete, table: String) extends IDeleteFrom {
        private[delete] def _buildFrom(cl: ConsistencyLevel) = {
          val res = delete._buildDeleteClause.from(table)
          res.setConsistencyLevel(cl)
          res
        }

        def USING(clauses: UsingClause*) = new DeleteUsing(this, clauses)
      }

      class DeleteUsing private[delete](deleteFrom: IDeleteFrom, using: Seq[UsingClause]) extends IDeleteFrom {
        private[delete] def _buildFrom(cl: ConsistencyLevel) = {
          val builder = deleteFrom._buildFrom(cl)
          using.foreach(clause => builder using clause.using)
          builder
        }
      }

      case class DeleteWhere private[delete](deleteFrom: IDeleteFrom, clauses: Seq[WhereClause]) extends BuildableToStatement[DeleteStatement] {
        def AND(clause: WhereClause) = copy(clauses = clauses :+ clause)

        def build(cl: ConsistencyLevel) = new DeleteStatement(clauses.foldLeft(deleteFrom._buildFrom(cl).where)(_ and _.clause))
      }

    }

    object batch {

      case class Batch(statements: Seq[BatcheableStatement]) extends BuildableToStatement[BatchStatement] {
        selfBatch =>
        private[batch] def _build(cl: ConsistencyLevel) = {
          val res = cqb.QueryBuilder.batch(statements.map(_.statement): _*)
          res.setConsistencyLevel(cl)
          res
        }

        def build(cl: ConsistencyLevel) = new BatchStatement(_build(cl))

        def USING(clauses: UsingClause*) = new BuildableToStatement[BatchStatement] {
          def build(cl: ConsistencyLevel) = {
            if (clauses.isEmpty) selfBatch.build(cl)
            else {
              new BatchStatement(clauses.drop(1).foldLeft(selfBatch._build(cl).using(clauses.head.using))(_ and _.using))
            }
          }
        }
      }

    }

    case class WhereClause(clause: cqb.Clause)

    case class IAssignment(assignment: cqb.Assignment)

    sealed trait UsingClause {
      def using: cqb.Using
    }

    case class TIMESTAMP(value: Long) extends UsingClause {
      def using = cqb.QueryBuilder.timestamp(value)
    }

    case class TTL(value: Int) extends UsingClause {
      def using = cqb.QueryBuilder.ttl(value)
    }

  }

  object SELECT {

    import internals.select._

    def apply(columns: Symbol*): ISelect = new Select(columns.map(_.name))

    def COUNT(all: ALL): ISelect = SelectCountAll

  }

  sealed trait ALL

  case object ALL extends ALL

  case object `*` extends ALL

  object INSERT {

    import internals.insert._

    def INTO(table: Symbol): InsertInto = new InsertInto(table.name)
  }

  object UPDATE {

    import internals.update._

    def apply(table: Symbol) = new UpdateTable(table.name)
  }

  object DELETE extends internals.delete.DeleteAll {

    import internals.delete._

    def apply(columns: Symbol*): IDelete = new Delete(columns.map(_.name): _*)
  }

  object BATCH {

    import internals.batch._

    def apply(first: BatcheableStatement, statements: BatcheableStatement*) = new Batch(first +: statements)

    def apply[T <% BatcheableStatement](statements: Seq[T]) = new Batch(statements.map(implicitly[T => BatcheableStatement]))
  }

  def TIMESTAMP(ts: Long) = new internals.TIMESTAMP(ts)

  def TTL(ttl: Int) = new internals.TTL(ttl)

  implicit class Column(val name: Symbol) extends AnyVal {
    def in(values: CQLValue*) = WhereClause(cqb.QueryBuilder.in(name.name, values.map(_.value): _*))

    def :=(value: CQLValue) = IAssignment(cqb.QueryBuilder.set(name.name, value.value))

    def ===(value: CQLValue) = WhereClause(cqb.QueryBuilder.eq(name.name, value.value))

    def >(value: CQLValue) = WhereClause(cqb.QueryBuilder.gt(name.name, value.value))

    def <(value: CQLValue) = WhereClause(cqb.QueryBuilder.lt(name.name, value.value))

    def >=(value: CQLValue) = WhereClause(cqb.QueryBuilder.gte(name.name, value.value))

    def <=(value: CQLValue) = WhereClause(cqb.QueryBuilder.lte(name.name, value.value))

    def asc = CQLOrdering(cqb.QueryBuilder.asc(name.name))

    def desc = CQLOrdering(cqb.QueryBuilder.desc(name.name))
  }

  implicit def buildableToStatementIsStatement[T <: Statement](bts: BuildableToStatement[T])(implicit cl: ConsistencyLevel = ConsistencyLevel.ONE): T = bts.build(cl)

  implicit def nativeStatementIsStatement(s: cd.Statement): Statement = new Statement(s)

  class Statement(val statement: cd.Statement)

  class SelectStatement(statement: cd.RegularStatement) extends Statement(statement)

  abstract class WriteStatement(statement: cd.Statement) extends Statement(statement)

  sealed abstract class BatcheableStatement(override val statement: cd.RegularStatement) extends WriteStatement(statement)

  class InsertStatement(statement: cd.RegularStatement) extends BatcheableStatement(statement)

  class UpdateStatement(statement: cd.RegularStatement) extends BatcheableStatement(statement)

  class DeleteStatement(statement: cd.RegularStatement) extends BatcheableStatement(statement)

  class BatchStatement(statement: cd.Statement) extends WriteStatement(statement)

}
