package ru.yandex.atom.service

import akka.actor.{Props, ActorRef, Actor}
import ru.yandex.atom.utils.actor.AtomActorLogging
import com.typesafe.config.Config
import scala.concurrent.duration.{Duration, FiniteDuration}
import akka.io.IO
import spray.can.Http
import spray.http._
import spray.http.Uri.{Path, Query}
import java.net.InetAddress
import spray.can.client.HostConnectorSettings
import akka.pattern.ask
import akka.util.Timeout
import spray.http.HttpRequest
import ru.yandex.atom.data.ReqID
import scala.Some
import spray.http.HttpResponse
import spray.json.{JsonParser, BasicFormats}
import scala.util.{Success, Failure}

/**
 * @author avhaliullin
 */
trait APIKeysActorComponentImpl extends APIKeysActorComponent {
  component =>

  class APIKeysActor(config: APIKeysConfig) extends Actor with AtomActorLogging with BasicFormats {

    import APIKeysRequest._
    import APIKeysResponse._

    implicit val ec = context.dispatcher

    val Code200 = StatusCode.int2StatusCode(200)
    val Code403 = StatusCode.int2StatusCode(403)
    val Code404 = StatusCode.int2StatusCode(404)

    override def preStart() = {
      implicit val actorSystem = context.system

      IO(Http) ! Http.HostConnectorSetup(host = config.host, port = config.port,
        settings = Some(HostConnectorSettings(actorSystem).copy(idleTimeout = Duration.Inf)))
    }

    override def receive = {
      case Http.HostConnectorInfo(connector, req) =>
        logWithId.info(ReqID("api keys", "init"), "Connector to API Keys initialized " + req)
        context.become(listen(connector))
    }

    def listen(connector: ActorRef): Receive = {
      case req@CheckKey(id, key, ip) =>
        val client = sender
        val addr = ip.toOption.getOrElse(InetAddress.getLoopbackAddress)
        val httpRequest = HttpRequest(uri = Uri(path = Path("/api/check_key"), query = Query(
          "service_token" -> config.serviceToken,
          "key" -> key,
          "user_ip" -> addr.getHostAddress,
          "ip_v" -> (if (addr.getHostAddress.contains(":")) "6" else "4")
        )))
        implicit val timeout = Timeout(config.timeout)
        (connector ? httpRequest)
          .mapTo[HttpResponse]
          .map {
          resp =>
            if (resp.status == Code403 || resp.status == Code404) {
              // Логируем, потому что адекватных текстовых кодов ошибок в API Keys пока нет,
              // а 404 может означать как "ключ не найден", так и "сервисный токен не найден"
              logWithId.info(req.id, "Auth failed: {}", resp.entity.asString)
              None: Option[Long]
            } else if (resp.status != Code200) {
              throw new RuntimeException(s"API Keys responded with code ${resp.status} and body ${resp.entity.asString}")
            } else {
              val json = JsonParser(resp.entity.asString)
              val uidOption = for {
                keyInfo <- json.asJsObject().fields.get("key_info")
                userInfo <- keyInfo.asJsObject().fields.get("user")
                userIdNode <- userInfo.asJsObject().fields.get("uid")
              } yield userIdNode.convertTo[Long]
              Some(uidOption.getOrElse(
                throw new RuntimeException("No uid found in API Keys response " + resp.entity.asString)))
            }
        }.onComplete {
          case Success(uidOpt) =>
            client ! KeyChecked(req, uidOpt)
            connector ! HttpRequest(uri = Uri(path = Path("/api/update_counters"), query = Query(
              "service_token" -> config.serviceToken,
              "key" -> key
            )))
          case Failure(e) =>
            logWithId.error(req.id, e, "API Keys request failed")
            client ! FailureResponse(req, e)
        }

      case req@CreateKey(id, uid) =>
        val client = sender
        val httpRequest = HttpRequest(uri = Uri(path = Path("/api/create_key"), query = Query(
          "service_token" -> config.serviceToken,
          "user_uid" -> uid.toString
        )))
        implicit val timeout = Timeout(config.timeout)
        (connector ? httpRequest)
          .mapTo[HttpResponse]
          .map {
          resp =>
            if (resp.status != Code200) {
              throw new RuntimeException(s"API keys responded with code ${resp.status} and body ${resp.entity.asString}")
            } else {
              val json = JsonParser(resp.entity.asString)
              val uidOption = for {
                result <- json.asJsObject().fields.get("result")
                key <- result.asJsObject().fields.get("key")
              } yield key.convertTo[String]
              uidOption.getOrElse(throw new RuntimeException("No key found in API Keys response " + resp.entity.asString))
            }
        }.onComplete {
          case Success(key) =>
            client ! KeyCreated(req, key)
          case Failure(e) =>
            logWithId.error(req.id, e, "API Keys request failed")
            client ! FailureResponse(req, e)
        }
    }
  }

  object APIKeysActor {
    def props(config: APIKeysConfig) = Props(classOf[APIKeysActor], component, config)
  }

  object APIKeysConfig {

    import ru.yandex.atom.utils.config._

    implicit def apply(config: Config): APIKeysConfig = new APIKeysConfig(
      host = config.getString("host"),
      port = config.getInt("port"),
      serviceToken = config.getString("serviceToken"),
      timeout = config.getFiniteDuration("timeout")
    )
  }

  case class APIKeysConfig(host: String, port: Int, serviceToken: String, timeout: FiniteDuration)

}
