package ru.yandex.atom.service

import akka.actor.ActorRef
import ru.yandex.atom.error.{InternalProblem, UserProblem}
import ru.yandex.atom.utils.actor.messages._
import ru.yandex.atom.data.{ReqID, URLList}

/**
 * @author avhaliullin
 */
trait UrlListsActorComponent {
  def urlListsActor: ActorRef

  sealed trait UrlListsRequest extends AbstractMessage {
    def id: ReqID
  }

  object UrlListsRequest {

    case class CreateUrls(id: ReqID, userId: Long, urls: Set[String]) extends UrlListsRequest

    case class UpdateUrls(id: ReqID, userId: Long, listId: Long, list: Set[String]) extends UrlListsRequest

    case class ViewUrls(id: ReqID, userId: Long, listId: Long) extends UrlListsRequest

    case class ListIds(id: ReqID, userId: Long) extends UrlListsRequest

  }

  sealed trait UrlListsResponse extends ResponseMessage[UrlListsRequest]

  object UrlListsResponse {

    case class ViewUrlsResponse(request: UrlListsRequest.ViewUrls, list: URLList) extends UrlListsResponse
    with SetReqType[UrlListsRequest.ViewUrls]

    case class CreateUrlsResponse(request: UrlListsRequest.CreateUrls, listId: Long) extends UrlListsResponse
    with SetReqType[UrlListsRequest.CreateUrls]

    case class UpdateUrlsResponse(request: UrlListsRequest.UpdateUrls) extends UrlListsResponse
    with SetReqType[UrlListsRequest.UpdateUrls]

    case class ListIdsResponse(request: UrlListsRequest.ListIds, ids: Set[Long]) extends UrlListsResponse
    with SetReqType[UrlListsRequest.ListIds]

    sealed trait FailureResponse extends UrlListsResponse with SetReqType[UrlListsRequest] {
      self: IFailureResponse =>
    }

    object FailureResponse {
      def apply(req: UrlListsRequest, cause: IFailureMagnet): FailureResponse = {
        cause match {
          case IFailureMagnet.InternalProblemFailure(problem) => InternalProblemResponse(req, problem)
          case IFailureMagnet.UserProblemFailure(problem) => UserProblemResponse(req, problem)
        }
      }
    }

    case class UserProblemResponse(request: UrlListsRequest, problem: UserProblem) extends FailureResponse with IUserProblemResponse

    case class InternalProblemResponse(request: UrlListsRequest, problem: InternalProblem) extends FailureResponse with IInternalProblemResponse

  }

}
