package ru.yandex.atom.utils.concurrent

import scala.concurrent.{ExecutionContext, Future}

/**
 * @author avhaliullin
 */

sealed abstract class AsyncStream[+T] {
  def head: T

  def tail: Future[AsyncStream[T]]

  def isEmpty: Boolean

  def foldLeft[R](first: R)(f: (R, T) => R)(implicit ec: ExecutionContext): Future[R]
}

object AsyncStream {

  def apply[T](t: T, tail: => Future[AsyncStream[T]]): AsyncStream[T] = new Cons[T](t, tail)

  def apply[T](t: T): AsyncStream[T] = new Cons[T](t, Future.successful(Empty))

  class Cons[+T](val head: T, tl: => Future[AsyncStream[T]]) extends AsyncStream[T] {
    lazy val _tail = tl

    def tail = _tail

    val isEmpty = false

    override def foldLeft[R](first: R)(f: (R, T) => R)(implicit ec: ExecutionContext) = {
      tail.flatMap(_.foldLeft(f(first, head))(f))
    }
  }

  object Cons {
    def apply[T](head: T, tail: => Future[AsyncStream[T]]): Cons[T] = new Cons(head, tail)
  }

  case object Empty extends AsyncStream[Nothing] {
    override def head = throw new NoSuchElementException("head of empty stream")

    override def tail = throw new UnsupportedOperationException("tail of empty stream")

    override val isEmpty = true

    override def foldLeft[R](first: R)(f: (R, Nothing) => R)(implicit ec: ExecutionContext) = Future.successful(first)
  }

}
