package ru.yandex.atom.utils.log

import akka.event.{DiagnosticLoggingAdapter, NoLogging}
import ru.yandex.atom.data.ReqID
import org.slf4j.{MDC, LoggerFactory, Logger}
import scala.reflect.ClassTag

/**
 * @author avhaliullin
 */
trait AtomLogger {
  def error(id: ReqID, cause: Throwable, message: String): Unit

  def error(id: ReqID, message: String): Unit

  def error(id: ReqID, template: String, arg1: Any): Unit

  def error(id: ReqID, template: String, arg1: Any, arg2: Any): Unit

  def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit

  def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit

  def warning(id: ReqID, message: String): Unit

  def warning(id: ReqID, template: String, arg1: Any): Unit

  def warning(id: ReqID, template: String, arg1: Any, arg2: Any): Unit

  def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit

  def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit

  def info(id: ReqID, message: String): Unit

  def info(id: ReqID, template: String, arg1: Any): Unit

  def info(id: ReqID, template: String, arg1: Any, arg2: Any): Unit

  def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit

  def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit

  def debug(id: ReqID, message: String): Unit

  def debug(id: ReqID, template: String, arg1: Any): Unit

  def debug(id: ReqID, template: String, arg1: Any, arg2: Any): Unit

  def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit

  def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit
}

object AtomLogger {

  def apply(log: DiagnosticLoggingAdapter): AtomLogger = new AkkaAtomLogger(log)

  def apply(log: Logger): AtomLogger = new Slf4jAtomLogger(log)

  def apply[T: ClassTag](): AtomLogger = AtomLogger(LoggerFactory.getLogger(implicitly[ClassTag[T]].runtimeClass))

  class AkkaAtomLogger(val log: DiagnosticLoggingAdapter) extends AtomLogger {
    def error(id: ReqID, cause: Throwable, message: String): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(cause, message)
        log.clearMDC()
      }
    }

    def error(id: ReqID, message: String): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(message)
        log.clearMDC()
      }
    }

    def error(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(template, arg1)
        log.clearMDC()
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(template, arg1, arg2)
        log.clearMDC()
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(template, arg1, arg2, arg3)
        log.clearMDC()
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isErrorEnabled) {
        log.mdc(Map("reqId" -> id))
        log.error(template, arg1, arg2, arg3, arg4)
        log.clearMDC()
      }
    }

    def warning(id: ReqID, message: String): Unit = {
      if (log.isWarningEnabled) {
        log.mdc(Map("reqId" -> id))
        log.warning(message)
        log.clearMDC()
      }
    }

    def warning(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isWarningEnabled) {
        log.mdc(Map("reqId" -> id))
        log.warning(template, arg1)
        log.clearMDC()
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isWarningEnabled) {
        log.mdc(Map("reqId" -> id))
        log.warning(template, arg1, arg2)
        log.clearMDC()
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isWarningEnabled) {
        log.mdc(Map("reqId" -> id))
        log.warning(template, arg1, arg2, arg3)
        log.clearMDC()
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isWarningEnabled) {
        log.mdc(Map("reqId" -> id))
        log.warning(template, arg1, arg2, arg3, arg4)
        log.clearMDC()
      }
    }

    def info(id: ReqID, message: String) {
      if (log.isInfoEnabled) {
        log.mdc(Map("reqId" -> id))
        log.info(message)
        log.clearMDC()
      }
    }

    def info(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isInfoEnabled) {
        log.mdc(Map("reqId" -> id))
        log.info(template, arg1)
        log.clearMDC()
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isInfoEnabled) {
        log.mdc(Map("reqId" -> id))
        log.info(template, arg1, arg2)
        log.clearMDC()
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isInfoEnabled) {
        log.mdc(Map("reqId" -> id))
        log.info(template, arg1, arg2, arg3)
        log.clearMDC()
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isInfoEnabled) {
        log.mdc(Map("reqId" -> id))
        log.info(template, arg1, arg2, arg3, arg4)
        log.clearMDC()
      }
    }

    def debug(id: ReqID, message: String) {
      if (log.isDebugEnabled) {
        log.mdc(Map("reqId" -> id))
        log.debug(message)
        log.clearMDC()
      }
    }

    def debug(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isDebugEnabled) {
        log.mdc(Map("reqId" -> id))
        log.debug(template, arg1)
        log.clearMDC()
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isDebugEnabled) {
        log.mdc(Map("reqId" -> id))
        log.debug(template, arg1, arg2)
        log.clearMDC()
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isDebugEnabled) {
        log.mdc(Map("reqId" -> id))
        log.debug(template, arg1, arg2, arg3)
        log.clearMDC()
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isDebugEnabled) {
        log.mdc(Map("reqId" -> id))
        log.debug(template, arg1, arg2, arg3, arg4)
        log.clearMDC()
      }
    }
  }

  class Slf4jAtomLogger(log: Logger) extends AtomLogger {

    // slf4j-логгер кушает Object's (== AnyRef), а надо бы передавать Any. Поэтому проще запользовать форматтер из акки
    private def format(s: String, args: Any*): String = NoLogging.format(s, args: _*)

    def error(id: ReqID, cause: Throwable, message: String): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(message, cause)
        MDC.remove("reqId")
      }
    }

    def error(id: ReqID, message: String): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(message)
        MDC.remove("reqId")
      }
    }

    def error(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(format(template, arg1))
        MDC.remove("reqId")
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(format(template, arg1, arg2))
        MDC.remove("reqId")
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(format(template, arg1, arg2, arg3))
        MDC.remove("reqId")
      }
    }

    def error(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isErrorEnabled) {
        MDC.put("reqId", id.toString)
        log.error(format(template, arg1, arg2, arg3, arg4))
        MDC.remove("reqId")
      }
    }

    def warning(id: ReqID, message: String): Unit = {
      if (log.isWarnEnabled) {
        MDC.put("reqId", id.toString)
        log.warn(message)
        MDC.remove("reqId")
      }
    }

    def warning(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isWarnEnabled) {
        MDC.put("reqId", id.toString)
        log.warn(format(template, arg1))
        MDC.remove("reqId")
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isWarnEnabled) {
        MDC.put("reqId", id.toString)
        log.warn(format(template, arg1, arg2))
        MDC.remove("reqId")
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isWarnEnabled) {
        MDC.put("reqId", id.toString)
        log.warn(format(template, arg1, arg2, arg3))
        MDC.remove("reqId")
      }
    }

    def warning(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isWarnEnabled) {
        MDC.put("reqId", id.toString)
        log.warn(format(template, arg1, arg2, arg3, arg4))
        MDC.remove("reqId")
      }
    }

    def info(id: ReqID, message: String) {
      if (log.isInfoEnabled) {
        MDC.put("reqId", id.toString)
        log.info(message)
        MDC.remove("reqId")
      }
    }

    def info(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isInfoEnabled) {
        MDC.put("reqId", id.toString)
        log.info(format(template, arg1))
        MDC.remove("reqId")
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isInfoEnabled) {
        MDC.put("reqId", id.toString)
        log.info(format(template, arg1, arg2))
        MDC.remove("reqId")
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isInfoEnabled) {
        MDC.put("reqId", id.toString)
        log.info(format(template, arg1, arg2, arg3))
        MDC.remove("reqId")
      }
    }

    def info(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isInfoEnabled) {
        MDC.put("reqId", id.toString)
        log.info(format(template, arg1, arg2, arg3, arg4))
        MDC.remove("reqId")
      }
    }

    def debug(id: ReqID, message: String) {
      if (log.isDebugEnabled) {
        MDC.put("reqId", id.toString)
        log.debug(message)
        MDC.remove("reqId")
      }
    }

    def debug(id: ReqID, template: String, arg1: Any): Unit = {
      if (log.isDebugEnabled) {
        MDC.put("reqId", id.toString)
        log.debug(format(template, arg1))
        MDC.remove("reqId")
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any): Unit = {
      if (log.isDebugEnabled) {
        MDC.put("reqId", id.toString)
        log.debug(format(template, arg1, arg2))
        MDC.remove("reqId")
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = {
      if (log.isDebugEnabled) {
        MDC.put("reqId", id.toString)
        log.debug(format(template, arg1, arg2, arg3))
        MDC.remove("reqId")
      }
    }

    def debug(id: ReqID, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = {
      if (log.isDebugEnabled) {
        MDC.put("reqId", id.toString)
        log.debug(format(template, arg1, arg2, arg3, arg4))
        MDC.remove("reqId")
      }
    }
  }

}
  
