package ru.yandex.atom.periodic

import ru.yandex.atom.zookeeper.{Stat, ZookeeperActorComponent}
import org.apache.zookeeper.{CreateMode, KeeperException}
import akka.pattern.ask
import ru.yandex.atom.data.ReqID
import ru.yandex.atom.error.ErrorUtils
import scala.concurrent.Await
import java.nio.ByteBuffer
import akka.util.Timeout

/**
 * @author avhaliullin
 */
trait ZookeeperSyncUtils {
  self: ZookeeperActorComponent =>

  object ZK {
    def children[T](id: ReqID, path: String, onSuccess: (Stat, Seq[String]) => T)(implicit timeout: Timeout): T = {
      childrenRecover(id, path, onSuccess)(PartialFunction.empty)
    }

    def childrenRecover[T](id: ReqID, path: String, onSuccess: (Stat, Seq[String]) => T)
                          (onFailure: PartialFunction[KeeperException.Code, T])
                          (implicit timeout: Timeout): T = {
      try {
        val res = Await.result(ErrorUtils.safeMapFutureMessage[ZookeeperResponse, ZookeeperResponse.ChildrenResponse](
          zookeeperActor ? ZookeeperRequest.ChildrenRequest(id, path)), timeout.duration)
        onSuccess(res.stat, res.children)
      } catch {
        case err: KeeperException if onFailure.isDefinedAt(err.code()) =>
          onFailure(err.code())
      }
    }

    def delete(id: ReqID, path: String, version: Int)(implicit timeout: Timeout): Unit = {
      deleteRecover(id, path, version, () => ())(PartialFunction.empty)
    }

    def deleteRecover[T](id: ReqID, path: String, version: Int, onSuccess: () => T)
                        (onFailure: PartialFunction[KeeperException.Code, T])
                        (implicit timeout: Timeout): T = {
      try {
        val res = Await.result(ErrorUtils.safeMapFutureMessage[ZookeeperResponse, ZookeeperResponse.DeleteResponse](
          zookeeperActor ? ZookeeperRequest.DeleteRequest(id, path, version)), timeout.duration)
        onSuccess()
      } catch {
        case err: KeeperException if onFailure.isDefinedAt(err.code()) =>
          onFailure(err.code())
      }
    }

    def update[T](id: ReqID, path: String, data: Array[Byte], version: Int, onSuccess: Stat => T)
                 (implicit timeout: Timeout): T = {
      updateRecover(id, path, data, version, onSuccess)(PartialFunction.empty)
    }

    def updateRecover[T](id: ReqID, path: String, data: Array[Byte], version: Int, onSuccess: Stat => T)
                        (onFailure: PartialFunction[KeeperException.Code, T])
                        (implicit timeout: Timeout): T = {
      try {
        val res = Await.result(ErrorUtils.safeMapFutureMessage[ZookeeperResponse, ZookeeperResponse.UpdateResponse](
          zookeeperActor ? ZookeeperRequest.UpdateRequest(id, path, data, version)), timeout.duration)
        onSuccess(res.stat)
      } catch {
        case err: KeeperException if onFailure.isDefinedAt(err.code()) =>
          onFailure(err.code())
      }
    }

    def create(id: ReqID, path: String, data: Array[Byte], mode: CreateMode)(implicit timeout: Timeout): Unit = {
      createRecover(id, path, data, mode, () => ())(PartialFunction.empty)
    }

    def createRecover[T](id: ReqID, path: String, data: Array[Byte], mode: CreateMode, onSuccess: () => T)(onFailure: PartialFunction[KeeperException.Code, T])
                        (implicit timeout: Timeout): T = {
      try {
        Await.result(ErrorUtils.safeMapFutureMessage[ZookeeperResponse, ZookeeperResponse.CreateResponse](
          zookeeperActor ? ZookeeperRequest.CreateRequest(id, path, data, mode)), timeout.duration)
        onSuccess()
      } catch {
        case err: KeeperException if onFailure.isDefinedAt(err.code()) =>
          onFailure(err.code())
      }
    }

    def read[T](id: ReqID, path: String, f: (Stat, Array[Byte]) => T)(implicit timeout: Timeout): T = {
      readRecover(id, path, f)(PartialFunction.empty)
    }

    def readRecover[T](id: ReqID, path: String, onSuccess: (Stat, Array[Byte]) => T)(onFailure: PartialFunction[KeeperException.Code, T])
                      (implicit timeout: Timeout): T = {
      try {
        val res = Await.result(ErrorUtils.safeMapFutureMessage[ZookeeperResponse, ZookeeperResponse.ReadResponse](
          zookeeperActor ? ZookeeperRequest.ReadRequest(id, path)), timeout.duration)
        onSuccess(res.stat, res.data)
      } catch {
        case err: KeeperException if onFailure.isDefinedAt(err.code()) =>
          onFailure(err.code())
      }
    }

    def serializeString(s: String): Array[Byte] = s.getBytes("UTF-8")

    def deserializeString(a: Array[Byte]) = new String(a, "UTF-8")

    def serializeLong(timeMs: Long): Array[Byte] = {
      val res = ByteBuffer.allocate(8)
      res.putLong(timeMs)
      res.array()
    }

    def deserializeLong(arr: Array[Byte]): Long = {
      ByteBuffer.wrap(arr).getLong
    }

  }

}
