package ru.yandex.atom.frontend.error

import spray.json._
import spray.httpx.marshalling.{BasicMarshallers, ToResponseMarshallingContext, ToResponseMarshaller}
import ru.yandex.atom.error.{InternalException, InternalProblem, UserProblem, UserException}
import spray.http.ContentTypes
import ru.yandex.atom.error.UserProblem.{MalformedRequest, UrlListNotFound, InvalidUrlListSize, MalformedUrls}
import ru.yandex.atom.error.InternalProblem.{UnknownProblem, TemporarilyProblem}

/**
 * @author avhaliullin
 */
object JsonErrorMarshalling extends DefaultJsonProtocol {

  private def userProblemDesc(p: UserProblem): ErrorDesc = {
    p match {
      case UrlListNotFound(userId, id) => ErrorDesc.UrlListNotFound(id)
      case InvalidUrlListSize(size) => ErrorDesc.InvalidListSize(size)
      case MalformedUrls(urls) => ErrorDesc.MalformedUrls(urls)
      case MalformedRequest(desc) => ErrorDesc.ApplicationBadRequest(desc)
    }
  }

  private def internalProblemDesc(p: InternalProblem): ErrorDesc = {
    p match {
      case TemporarilyProblem => ErrorDesc.ServiceUnavailable()
      case UnknownProblem => ErrorDesc.InternalServerError()
    }
  }

  private def throwableDesc(t: Throwable): ErrorDesc = {
    t match {
      case UserException(problem) => userProblemDesc(problem)
      case InternalException(problem) => internalProblemDesc(problem)
      case x => internalProblemDesc(InternalProblem.throwable2InternalProblem(x))
    }
  }

  implicit def exceptionMarshaller(implicit errorDescMarshaller: ToResponseMarshaller[ErrorDesc]): ToResponseMarshaller[Throwable] = {
    new ToResponseMarshaller[Throwable] {
      def apply(t: Throwable, ctx: ToResponseMarshallingContext) {
        val desc = throwableDesc(t)
        implicitly[ToResponseMarshaller[ErrorDesc]].apply(desc, ctx)
      }
    }
  }

  implicit def errorDescMarshaller(implicit printer: JsonPrinter = PrettyPrinter): ToResponseMarshaller[ErrorDesc] = {
    new ToResponseMarshaller[ErrorDesc] {
      def apply(desc: ErrorDesc, ctx: ToResponseMarshallingContext) {
        val json = JsObject("errors" -> JsArray(JsObject(
          "code" -> JsString(desc.errorCode),
          "message" -> JsString(desc.message),
          "attributes" -> desc.params
        )))
        val jsonString = printer(json)

        (BasicMarshallers.stringMarshaller(ContentTypes.`application/json`): ToResponseMarshaller[String]
          )(jsonString, ctx.withResponseMapped(_.copy(status = desc.status).withHeaders(desc.headers)))
      }
    }
  }


}
