package ru.yandex.autodoc.common.out.json;

import ru.yandex.autodoc.common.out.json.builder.*;

import java.text.DateFormat;
import java.util.Date;

/**
 * @author avhaliullin
 */
public class JsonConvertableUtils {
    private static final String DEFAULT_PRETTY_INDENT = "   ";

    public static String compactPrint(JsonValueWriter writer) {
        StringBuilder sb = new StringBuilder();
        JsonValueBuilder<ContainerBuilder> builder = createValueBuilder(sb);
        writer.writeValue(builder);
        return sb.toString();
    }

    public static JsonValueBuilder<ContainerBuilder> createValueBuilder(Appendable appendable) {
        return JsonValueBuilder.createRootBuilder(new PureJsonAppendable(appendable));
    }

    public static JsonValueBuilder<ContainerBuilder> createCommentedValueBuilder(Appendable appendable) {
        return createCommentedValueBuilder(appendable, DEFAULT_PRETTY_INDENT);
    }

    public static JsonValueBuilder<ContainerBuilder> createCommentedValueBuilder(Appendable appendable, String indent) {
        return JsonValueBuilder.createRootBuilder(new CommentedJsonAppendable(appendable, indent));
    }


    public static <T> JsonArrayWriter makeArrayWriter(final Iterable<T> collection, final JsonValueWriterFactory<T> writerFactory) {
        return new JsonArrayWriter() {
            @Override
            public <C extends ContainerBuilder> JsonArrayBuilder<C> writeArray(JsonArrayBuilder<C> builder) {
                for (T item : collection) {
                    builder = writerFactory.writeAsJsonValue(item).writeValue(builder.element());
                }
                return builder;
            }
        };
    }

    public static JsonArrayWriter makeArrayWriter(final Iterable<? extends JsonValueWriter> items) {
        return new JsonArrayWriter() {
            @Override
            public <C extends ContainerBuilder> JsonArrayBuilder<C> writeArray(JsonArrayBuilder<C> builder) {
                for (JsonValueWriter writer : items) {
                    builder = writer.writeValue(builder.element());
                }
                return builder;
            }
        };
    }

    public static final JsonValueWriterFactory<String> STRING_WRITER_FACTORY = new JsonValueWriterFactory<String>() {
        @Override
        public JsonValueWriter writeAsJsonValue(final String data) {
            return new JsonValueWriter() {
                @Override
                public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                    return builder.value(data);
                }
            };
        }
    };

    public static JsonValueWriterFactory<Date> oldDateWriterFactory(final DateFormat format) {
        return new JsonValueWriterFactory<Date>() {
            @Override
            public JsonValueWriter writeAsJsonValue(final Date data) {
                return new JsonValueWriter() {
                    @Override
                    public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                        return builder.valueDate(data, format);
                    }
                };
            }
        };
    }

    public static JsonValueWriter stringWriter(final String x) {
        return STRING_WRITER_FACTORY.writeAsJsonValue(x);
    }

    public static JsonValueWriter intWriter(final Integer x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueInt(x);
            }
        };
    }

    public static JsonValueWriter longWriter(final Long x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueLong(x);
            }
        };
    }

    public static JsonValueWriter booleanWriter(final Boolean x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueBoolean(x);
            }
        };
    }

    public static JsonValueWriter floatWriter(final Float x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueDouble(x == null ? null : x.doubleValue());
            }
        };
    }

    public static JsonValueWriter doubleWriter(final Double x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueDouble(x);
            }
        };
    }

    public static JsonValueWriter byteWriter(final Byte x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueInt(x == null ? null : x.intValue());
            }
        };
    }

    public static JsonValueWriter shortWriter(final Short x) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                return builder.valueInt(x == null ? null : x.intValue());
            }
        };
    }

    public static final JsonValueWriter NULL_WRITER = new JsonValueWriter() {
        @Override
        public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
            return builder.valueNull();
        }
    };

    public static final JsonValueWriter EMPTY_OBJECT = new JsonValueWriter() {
        @Override
        public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
            return builder.valueObject().endObject();
        }
    };
}
