package ru.yandex.autodoc.common.out.json.builder;

import ru.yandex.autodoc.common.out.json.*;
import ru.yandex.autodoc.common.util.enums.IEnumResolver;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author avhaliullin
 */
public class JsonValueBuilder<C extends ContainerBuilder> {
    private final JsonAppendable appendable;
    private final C container;
    private boolean used = false;
    public static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
    public static final DateFormat DATE_TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    public JsonValueBuilder(JsonAppendable appendable, C container) {
        this.container = container;
        this.appendable = appendable;
    }

    public C valueObjectAsString(Object value) {
        if (value == null) {
            return valueNull();
        } else {
            use();
            appendable.appendStringValue(value.toString());
            return container;
        }
    }

    public C value(String value) {
        return valueObjectAsString(value);
    }

    public <T> C valueEnum(T value, IEnumResolver<T> resolver) {
        C res;
        if (value == null) {
            res = valueNull();
        } else {
            res = value(resolver.getName(value));
        }
        appendable.commentForLastKey("One of [" + resolver.join(resolver.listAll(), ", ") + "]");
        return res;
    }

    public C valueInt(Integer value) {
        if (value == null) {
            return valueNull();
        } else {
            use();
            appendable.appendNumericValue(String.valueOf(value));
            return container;
        }
    }

    public C valueLong(Long value) {
        if (value == null) {
            return valueNull();
        } else {
            use();
            appendable.appendNumericValue(String.valueOf(value));
            return container;
        }
    }

    public C valueBoolean(Boolean value) {
        if (value == null) {
            return valueNull();
        } else {
            use();
            appendable.appendBooleanValue(value);
            return container;
        }
    }

    public C valueDouble(Double value, DecimalFormat format) {
        if (value == null) {
            return valueNull();
        } else {
            use();
            appendable.appendNumericValue(format.format(value));
            return container;
        }
    }

    public C valueDouble(Double value) {
        return valueDouble(value, new DecimalFormat());
    }

    public C valueDateTime(Date date) {
        if (date == null) {
            return valueNull();
        } else {
            return valueObject()
                    .keyValueLong("timestamp", date.getTime() / 1000)
                    .keyValue("dateTime", DATE_TIME_FORMAT.format(date))
                    .endObject();
        }
    }

    private static Date getMidnightTimestamp(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime((date == null) ? new Date() : date);

        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);

        return cal.getTime();
    }


    public C valueDate(Date date, DateFormat format) {
        if (date == null) {
            return valueNull();
        } else {
            return valueObject()
                    .keyValueLong("timestamp", getMidnightTimestamp(date).getTime() / 1000)
                    .keyValue("date", format.format(date))
                    .endObject();
        }
    }

    public C valueDate(Date date) {
        return valueDate(date, DATE_FORMAT);
    }

    public C valueNull() {
        use();
        appendable.appendNullValue();
        return container;
    }

    public C valueObjectName(final String name) {
        return valueObject()
                .comment(name)
                .endObject();
    }

    public C valueObject(JsonObjectWriter writer) {
        JsonObjectBuilder<C> builder = valueObject();
        JsonObjectBuilder<C> resBuilder = writer.writeObject(builder);
        if (builder != resBuilder) {
            throw new RuntimeException("Json writer returned wrong object builder!");
        }
        return resBuilder.endObject();
    }

    public C valueArrayOfStrings(Iterable<String> collection) {
        JsonArrayBuilder<C> arrayBuilder = valueArray();
        for (String s : collection) {
            arrayBuilder = arrayBuilder.element().value(s);
        }
        return arrayBuilder.endArray();
    }

    public C valueArray(JsonArrayWriter writer) {
        JsonArrayBuilder<C> builder = valueArray();
        JsonArrayBuilder<C> resBuilder = writer.writeArray(builder);
        if (builder != resBuilder) {
            throw new RuntimeException("Json writer returned wrong object builder!");
        }
        return resBuilder.endArray();
    }

    public <T> C valueArray(Iterable<T> items, JsonValueWriterFactory<T> wrapper) {
        return valueArray(JsonConvertableUtils.makeArrayWriter(items, wrapper));
    }

    public C valueEmptyArray() {
        return valueArray().endArray();
    }

    public C writeValue(JsonValueWriter writer) {
        return writer == null ? valueNull() : writer.writeValue(this);
    }

    public <T> JsonPolymorphicBuilder<C, T> polymorphic(IEnumResolver<T> er, String fieldName, String typeFieldComment) {
        use();
        List<String> comments = new ArrayList<>();
        if (typeFieldComment != null) {
            comments.add(typeFieldComment);
        }
        comments.add("One of [" + er.join(er.listAll(), ", ") + "]");
        return new JsonPolymorphicBuilder<>(container, appendable, er, fieldName, comments);
    }

    public JsonNoTypePolymorphicBuilder<C> polymorphic() {
        use();
        return new JsonNoTypePolymorphicBuilder<>(container, appendable);
    }

    public JsonObjectBuilder<C> valueObject() {
        use();
        return new JsonObjectBuilder<>(appendable, container, true, true);
    }

    public JsonArrayBuilder<C> valueArray() {
        use();
        return new JsonArrayBuilder<>(appendable, container);
    }

    public <T> C valueEnumSet(Set<T> values, IEnumResolver<T> resolver) {
        appendable.commentForLastKey("Some of [" + resolver.join(resolver.listAll(), ", ") + "]");
        JsonArrayBuilder<C> builder = valueArray();
        for (T value : values) {
            builder.element().value(resolver.getName(value));
        }
        return builder.endArray();
    }

    public void comment(String s) {
        appendable.commentForLastKey(s);
    }

    private void use() {
        if (used) {
            throw new RuntimeException("JsonValueBuilder is already used!");
        }
        used = true;
        container.focusReturn();
    }

    public static JsonValueBuilder<ContainerBuilder> createRootBuilder(JsonAppendable appendable) {
        return new JsonValueBuilder<>(appendable, ROOT);
    }

    private static final ContainerBuilder ROOT = new ContainerBuilder(null) {
        @Override
        void focusReturn() {
        }
    };
}
