package ru.yandex.autodoc.common.out.json.builder;

import ru.yandex.autodoc.common.util.enums.IEnumResolver;

/**
 * @author avhaliullin
 */
public class PureJsonAppendable extends AbstractAppender implements JsonAppendable {

    public PureJsonAppendable(Appendable appendable) {
        super(appendable);
    }

    protected void printKey(String key) {
        append("\"" + quoteString(key) + "\"");
    }

    protected void printColon() {
        append(":");
    }

    protected void printStringValue(String value) {
        append("\"" + quoteString(value) + "\"");
    }

    private void printPrimitiveValue(String value) {
        append(value);
    }

    protected void printBooleanValue(boolean value) {
        printPrimitiveValue(String.valueOf(value));
    }

    protected void printNumericValue(String value) {
        printPrimitiveValue(value);
    }

    protected void printNullValue() {
        printPrimitiveValue("null");
    }

    protected void printComma() {
        append(",");
    }

    protected void printStartObject() {
        append("{");
    }

    protected void printEndObject() {
        append("}");
    }

    protected void printStartArray() {
        append("[");
    }

    protected void printEndArray() {
        append("]");
    }

    @Override
    public void appendKey(String key) {
        printKey(key);
        printColon();
    }

    @Override
    public void appendStringValue(String value) {
        printStringValue(value);
    }

    @Override
    public void appendNumericValue(String value) {
        printNumericValue(value);
    }

    @Override
    public void appendBooleanValue(boolean value) {
        printBooleanValue(value);
    }

    @Override
    public void appendNullValue() {
        printNullValue();
    }

    @Override
    public void appendComma() {
        printComma();
    }

    @Override
    public void startObject() {
        printStartObject();
    }

    @Override
    public void endObject() {
        printEndObject();
    }

    @Override
    public void endMapObject() {
        printEndObject();
    }

    @Override
    public void startArray() {
        printStartArray();
    }

    @Override
    public void endArray() {
        printEndArray();
    }

    @Override
    public void comment(String s) {
    }

    @Override
    public void commentForLastKey(String s) {
    }

    @Override
    public <T> PolymorphicHandler<T> polymorphic(IEnumResolver<T> er, String fieldName) {
        return new PurePolymorphicHandler<>(er, fieldName);
    }

    protected String quoteString(String s) {
        StringBuilder sb = new StringBuilder(s.length() * 2);
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            String toAppend = null;
            switch (c) {
                case '"':
                    toAppend = "\\\"";
                    break;
                case '\\':
                    toAppend = "\\\\";
                    break;
                case '\b':
                    toAppend = "\\b";
                    break;
                case '\f':
                    toAppend = "\\f";
                    break;
                case '\n':
                    toAppend = "\\n";
                    break;
                case '\r':
                    toAppend = "\\r";
                    break;
                case '\t':
                    toAppend = "\\t";
                    break;
                default:
                    if (c <= '\u001f' || (c >= '\u007f' && c <= '\u009f')) {
                        toAppend = String.format("\\u%04x", (int) c);
                    } else {
                        toAppend = "" + c;
                    }
            }
            sb.append(toAppend);
        }
        return sb.toString();
    }

    @Override
    public NoTypePolymorphicHandler polymorphic() {
        return new PureNoTypePolymorphicHandler();
    }

    private class PureNoTypePolymorphicHandler implements NoTypePolymorphicHandler {
        private boolean finished = false;

        @Override
        public JsonAppendable withCase(String caseId) {
            if (!finished) {
                startObject();
                return PureJsonAppendable.this;
            } else {
                return STUB_APPENDABLE;
            }
        }

        @Override
        public void endCase() {
            if (!finished) {
                endObject();
                finished = true;
            }
        }

        @Override
        public JsonAppendable getParent() {
            return PureJsonAppendable.this;
        }
    }

    private class PurePolymorphicHandler<T> implements PolymorphicHandler<T> {
        private final String fieldName;
        private final IEnumResolver<T> er;
        private boolean finished = false;

        private PurePolymorphicHandler(IEnumResolver<T> er, String fieldName) {
            this.fieldName = fieldName;
            this.er = er;
        }

        @Override
        public JsonAppendable withCase(T tpe) {
            if (!finished) {
                startObject();
                appendKey(fieldName);
                appendStringValue(er.getName(tpe));
                return PureJsonAppendable.this;
            } else {
                return STUB_APPENDABLE;
            }
        }

        @Override
        public void endCase() {
            if (!finished) {
                endObject();
                finished = true;
            }
        }

        @Override
        public JsonAppendable getParent() {
            return PureJsonAppendable.this;
        }
    }

    private static final JsonAppendable STUB_APPENDABLE = new JsonAppendable() {
        @Override
        public void appendKey(String key) {
        }

        @Override
        public void appendStringValue(String value) {
        }

        @Override
        public void appendNumericValue(String value) {
        }

        @Override
        public void appendBooleanValue(boolean value) {
        }

        @Override
        public void appendNullValue() {
        }

        @Override
        public void appendComma() {
        }

        @Override
        public void startObject() {
        }

        @Override
        public void endObject() {
        }

        @Override
        public void endMapObject() {
        }

        @Override
        public void startArray() {
        }

        @Override
        public void endArray() {
        }

        @Override
        public void comment(String s) {
        }

        @Override
        public void commentForLastKey(String s) {
        }

        @Override
        public <T> PolymorphicHandler<T> polymorphic(IEnumResolver<T> er, String fieldName) {
            return new StubHandler<T>();
        }

        @Override
        public NoTypePolymorphicHandler polymorphic() {
            return STUB_NO_TYPE_HANDLER;
        }
    };

    private static NoTypePolymorphicHandler STUB_NO_TYPE_HANDLER = new NoTypePolymorphicHandler() {
        @Override
        public JsonAppendable getParent() {
            return STUB_APPENDABLE;
        }

        @Override
        public JsonAppendable withCase(String caseId) {
            return STUB_APPENDABLE;
        }

        @Override
        public void endCase() {
        }
    };

    private static class StubHandler<T> implements PolymorphicHandler<T> {
        @Override
        public JsonAppendable withCase(T tpe) {
            return STUB_APPENDABLE;
        }

        @Override
        public void endCase() {
        }

        @Override
        public JsonAppendable getParent() {
            return STUB_APPENDABLE;
        }
    }
}
