package ru.yandex.autodoc.common.util.enums;

import org.apache.commons.lang3.Validate;

import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;

/**
 * @author avhaliullin
 */
public abstract class AbstractEnumResolver<T extends Enum<T>, Self extends IEnumResolver<T>> implements IEnumResolver<T> {
    protected final Class<T> enumClass;
    protected final Map<String, T> name2Enum = new HashMap<>();
    protected final EnumSet<T> valuesSet;
    protected final T defaultValue;

    public AbstractEnumResolver(Class<T> enumClass, T defaultValue) {
        this(enumClass, defaultValue, EnumSet.allOf(enumClass));
    }

    public AbstractEnumResolver(Class<T> enumClass, T defaultValue, Collection<T> universe) {
        Validate.notNull(enumClass);

        this.enumClass = enumClass;
        this.defaultValue = defaultValue;

        this.valuesSet = EnumSet.copyOf(universe);

        for (T val : universe) {
            name2Enum.put(getName(val).toLowerCase(), val);
        }
    }

    @Override
    public T getByName(String name) {
        if (name == null) {
            return null;
        }
        return name2Enum.get(name.toLowerCase());
    }

    @Override
    public T getDefault() {
        return defaultValue;
    }

    @Override
    public Collection<T> listAll() {
        return valuesSet;
    }

    @Override
    public String join(Collection<T> values, String separator) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (T value : values) {
            if (!first) {
                sb.append(separator);
            }
            first = false;
            sb.append(getName(value));
        }
        return sb.toString();
    }

    public Self of(T... subset) {
        return create(enumClass, defaultValue, Arrays.asList(subset));
    }

    protected abstract Self create(Class<T> enumClass, T defaultValue, Collection<T> universe);
}
