package ru.yandex.autodoc.wmtools;

import ru.yandex.autodoc.common.doc.DocUtils;
import ru.yandex.autodoc.common.doc.ErrorsDocumentationBuilder;
import ru.yandex.autodoc.common.doc.annotation.Example;
import ru.yandex.autodoc.common.doc.annotation.ExampleConstructor;
import ru.yandex.autodoc.common.doc.error.ErrorDescription;
import ru.yandex.autodoc.common.doc.error.ErrorsBlockInfo;
import ru.yandex.autodoc.wmtools.errors.view.AnnotatedUserExceptionWrapper;
import ru.yandex.autodoc.common.out.json.JsonValueWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.wmtools.common.error.UserException;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Строит документацию по всем внуренним и пользовательским ошибкам
 * <p>Внутренние ошибки рисуются просто для того, чтобы ко всем из них можно было установить ключи в танкере</p>
 * <p>Пользовательские ошибки рисуются с описанием и форматом сериализации, сгруппированы по категориям</p>
 * <p>Для включения категории ошибок в документацию, необходимо
 * {@link SpringErrorsDocumentationBuilder#setUserErrorClasses(java.util.List)  передать}
 * в билдер класс ошибки, удовлетворяющий ряду требований.</p>
 * <ul>
 * <li>Ошибки должны наследоваться от {@link ru.yandex.wmtools.common.error.UserException}. Правда, имеет смысл
 * наследоваться сразу от {@link ru.yandex.autodoc.wmtools.errors.CommonUserException} - потому что тогда можно
 * декларировать в throws метода CommonUserException, и быть уверенным, что вылетать будет только документированные
 * ошибки</li>
 * <li>Класс ошибки должен быть абстрактным, все конструкторы должны быть приватными (или protected, если нужно).
 * Это требование не контролируется.</li>
 * <li>Для каждого кода ошибки в теле абстрактного класса объявляется public static НЕ abstract класс ошибки,
 * наследующийся от абстрактного родительского</li>
 * </ul>
 * Требования к конкретным классам ошибок (относящимся к конкретным кодам ошибок):
 * <ul>
 * <li>Как уже было сказано, должны быть объявлены в теле обобщенного абстрактного класса, с модификаторами
 * public static, но не abstract</li>
 * <li>Коды ошибок (их текстовые представления) должны быть уникальны - не должно быть двух классов ошибок с
 * одинаковой IUserProblem. А еще текстовое представление IUserProblem не должно совпадать ни с одним названием
 * InternalProblem</li>
 * <li>Если у ошибки есть дополнительные поля, которые нужно отобразить в сериализованном представлении, используется
 * аннотация {@link ru.yandex.autodoc.common.doc.annotation.JsonField} на public final поле класса исключения</li>
 * <li>Документация отрисовывает примеры сериализованных ошибок. Класс ошибки должен предоставлять пример инстанса
 * одним из двух способов:
 * <ol>
 * <li>Завести public static final поле класса с типом этого класса - присвоить туда пример объекта ошибки.
 * Поле <b>должно</b> быть проаннотировано как {@link ru.yandex.autodoc.common.doc.annotation.Example}</li>
 * <li>Завести public конструктор без аргументов, проаннотированный
 * {@link ru.yandex.autodoc.common.doc.annotation.ExampleConstructor}</li>. Этот способ имеет смысл использовать,
 * когда у ошибки нет дополнительных параметров
 * </ol>
 * </li>
 * <li>На класс ошибки нужно навесить аннотацию {@link ru.yandex.autodoc.common.doc.annotation.Description} с
 * описанием ошибки</li>
 * </ul>
 * <p>
 * Примеры можно посмотреть в {@link ru.yandex.autodoc.wmtools.errors.CommonUserException}
 * </p>
 *
 * @author avhaliullin
 */
public class SpringErrorsDocumentationBuilder implements ErrorsDocumentationBuilder {
    private static final Logger log = LoggerFactory.getLogger(SpringErrorsDocumentationBuilder.class);

    private final Set<String> errorCodes = new HashSet<>();
    private final List<ErrorsBlockInfo> userErrors = new ArrayList<>();
    private List<Class> userErrorClasses;

    public void init() {

        for (Class errorClass : userErrorClasses) {
            buildUserExceptions(errorClass);
        }
    }

    private void addErrorCode(String code) {
        if (errorCodes.contains(code)) {
            throw new RuntimeException("Error code " + code + " is duplicated");
        }
        errorCodes.add(code);
    }

    private void addUserException(UserException e, String desc, List<ErrorDescription> map) {
        JsonValueWriter valueWriter = new AnnotatedUserExceptionWrapper(e);

        addErrorCode(e.getProblem().toString());

        map.add(new ErrorDescription(e.getProblem().toString(), desc, valueWriter));
    }

    private void buildUserExceptions(Class errorClass) {
        // Расширенные ошибки - со всякими там параметрами
        try {
            String description = DocUtils.getDescriptionForAnnotatedElement(errorClass);
            List<ErrorDescription> errorsBlock = new ArrayList<>();
            Class<?>[] classes = errorClass.getDeclaredClasses();
            for (Class<?> clazz : classes) {
                if (Modifier.isStatic(clazz.getModifiers()) &&
                        Modifier.isPublic(clazz.getModifiers()) &&
                        !Modifier.isInterface(clazz.getModifiers()) &&
                        !Modifier.isAbstract(clazz.getModifiers()) &&
                        UserException.class.isAssignableFrom(clazz)) {
                    Field[] fields = clazz.getFields();
                    UserException example = null;
                    for (Field field : fields) {
                        if (example == null &&
                                field.getAnnotation(Example.class) != null &&
                                Modifier.isStatic(field.getModifiers()) &&
                                Modifier.isPublic(field.getModifiers()) &&
                                clazz.isAssignableFrom(field.getType())
                                ) {
                            example = (UserException) field.get(null);
                        }
                    }
                    if (example == null) {

                        for (Constructor constructor : clazz.getDeclaredConstructors()) {
                            if (example == null &&
                                    constructor.getAnnotation(ExampleConstructor.class) != null &&
                                    Modifier.isPublic(constructor.getModifiers())) {
                                if (constructor.getParameterTypes().length > 0) {
                                    throw new RuntimeException("ExampleConstructor should not have arguments! Class " + clazz.getName());
                                }
                                example = (UserException) constructor.newInstance();
                            }
                        }
                    }
                    if (example == null) {
                        log.error("Neither of Example or ExampleConstructor provided for UserException: " + clazz.getName());
                    } else {
                        String desc = DocUtils.getDescriptionForAnnotatedElement(clazz);
                        addUserException(example, desc, errorsBlock);
                    }
                }
            }
            userErrors.add(new ErrorsBlockInfo(errorClass.getSimpleName(), description, errorsBlock));
        } catch (IllegalAccessException | InstantiationException | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    public List<ErrorsBlockInfo> getUserErrors() {
        return userErrors;
    }

    @Required
    public void setUserErrorClasses(List<Class> userErrorClasses) {
        this.userErrorClasses = userErrorClasses;
    }
}
