package ru.yandex.autodoc.wmtools;

import ru.yandex.autodoc.common.doc.DocUtils;
import ru.yandex.autodoc.common.doc.MethodsDocumentationBuilder;
import ru.yandex.autodoc.common.doc.abstracts.MethodDocumentation;
import ru.yandex.autodoc.common.doc.abstracts.ServantletWithDocumentation;
import ru.yandex.autodoc.common.doc.error.ErrorDescription;
import ru.yandex.autodoc.common.doc.result.MethodResult;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import ru.yandex.autodoc.common.doc.abstracts.ServantletOptionalDoc;
import ru.yandex.autodoc.common.doc.params.ParamDescriptor;
import ru.yandex.autodoc.common.doc.view.Markup;

import java.util.*;

/**
 * Строит документацию по методам из Spring'ового контекста, которые имплементят
 * {@link ru.yandex.autodoc.common.doc.abstracts.ServantletWithDocumentation}. Имя метода выбирается как имя бина,
 * с отрезанным суффиксом 'Servantlet'.
 * <p>Текстовое описание и категория ручки берутся из {@link ru.yandex.autodoc.common.doc.abstracts.ServantletOptionalDoc},
 * если ручка его имплементит, или из аннотаций {@link ru.yandex.autodoc.common.doc.annotation.Description} и
 * {@link ru.yandex.autodoc.common.doc.annotation.Category} соответственно. Аннотации имеют приоритет <i>(может быть, зря?).</i>
 * </p>
 * <p>Аннотация {@link ru.yandex.autodoc.common.doc.annotation.Category} также может быть навешана на пакет,
 * в котором лежит ручка - это способ указания категории с наименьшим приоритетом.</p>
 *
 * @author avhaliullin
 */
public class SpringMethodsDocumentationBuilder implements MethodsDocumentationBuilder, ApplicationContextAware {
    private final List<MethodDocumentation> methods = new ArrayList<>();
    private Set<String> contextsBlackList = null;
    private Set<String> contextsWhiteList = null;

    private boolean isAllowed(ServantletWithDocumentation bean) {
        if (contextsWhiteList != null) {
            return contextsWhiteList.contains(bean.definedInContext());
        } else if (contextsBlackList != null) {
            return !contextsBlackList.contains(bean.definedInContext());
        } else {
            return true;
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, ServantletWithDocumentation> beans = applicationContext.getBeansOfType(ServantletWithDocumentation.class);
        for (Map.Entry<String, ServantletWithDocumentation> entry : beans.entrySet()) {
            final String name = entry.getKey().substring(0, entry.getKey().length() - "Servantlet".length());
            final ServantletWithDocumentation bean = entry.getValue();

            if (!isAllowed(bean)) {
                continue;
            }
            final Class<?> clazz = bean.getClass();
            String documentation;
            String category;
            if (bean instanceof ServantletOptionalDoc) {
                ServantletOptionalDoc doc = (ServantletOptionalDoc) bean;
                documentation = doc.getDocumentation();
                category = doc.getCategory();
            } else {
                category = DocUtils.getCategoryForAnnotatedElement(clazz);
                documentation = DocUtils.getDescriptionForAnnotatedElement(clazz);
            }

            if (category == null) {
                category = DocUtils.getCategoryForAnnotatedElement(clazz.getPackage());
            }

            final String docFin = documentation;
            final String catFin = category;
            final boolean deprecated = clazz.isAnnotationPresent(Deprecated.class);

            methods.add(new MethodDocumentation() {
                @Override
                public String getName() {
                    return name;
                }

                @Override
                public String getCategory() {
                    return catFin;
                }

                @Override
                public boolean isDeprecated() {
                    return deprecated;
                }


                @Override
                public List<ParamDescriptor> getParamDescriptors() {
                    return bean.getParamDescriptors();
                }

                @Override
                public String getParamsStringDescription() {
                    return null;
                }

                @Override
                public String getDocumentation() {
                    return docFin;
                }

                @Override
                public MethodResult getMethodResult() {
                    return bean.getMethodResult();
                }

                @Override
                public boolean needAuthorization() {
                    return bean.needAuthorization();
                }

                @Override
                public String definedInContext() {
                    return bean.definedInContext();
                }

                @Override
                public List<ErrorDescription> getMethodErrors() {
                    return null;
                }

                @Override
                public Map<String, Markup> extraInfoBlocks() {
                    return Collections.emptyMap();
                }
            });
        }
    }

    public List<MethodDocumentation> getDocumentation() {
        return methods;
    }

    public void setContextsBlackList(Set<String> contextsBlackList) {
        this.contextsBlackList = contextsBlackList;
    }

    public void setContextsWhiteList(Set<String> contextsWhiteList) {
        this.contextsWhiteList = contextsWhiteList;
    }
}
