package ru.yandex.autodoc.wmtools.errors.view;

import ru.yandex.autodoc.common.doc.annotation.JsonField;
import ru.yandex.autodoc.common.out.json.builder.ContainerBuilder;
import ru.yandex.autodoc.common.out.json.builder.JsonArrayBuilder;
import ru.yandex.autodoc.common.out.json.builder.JsonObjectBuilder;
import ru.yandex.autodoc.common.out.json.builder.JsonValueBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.autodoc.wmtools.out.json.JsonObjectWrapper;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.common.util.xml.XmlConvertable;
import ru.yandex.wmtools.common.error.CodeErrorInfo;
import ru.yandex.wmtools.common.error.UserException;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

/**
 * @author avhaliullin
 */
public class AnnotatedUserExceptionWrapper extends JsonObjectWrapper<UserException> implements WrappedError, XmlConvertable {
    private static final Logger log = LoggerFactory.getLogger(AnnotatedUserExceptionWrapper.class);

    public AnnotatedUserExceptionWrapper(UserException data) {
        super(data);
    }

    private <C extends ContainerBuilder> C wrapUnknown(JsonValueBuilder<C> builder, Object obj) {
        if (obj == null) {
            return builder.valueNull();
        }
        if (obj instanceof String) {
            String s = (String) obj;
            return builder.value(s);
        }
        if (obj instanceof Integer) {
            Integer x = (Integer) obj;
            return builder.valueInt(x);
        }
        if (obj instanceof Long) {
            Long x = (Long) obj;
            return builder.valueLong(x);
        }
        if (obj instanceof Iterable) {
            JsonArrayBuilder<C> arrayBuilder = builder.valueArray();
            for (Object element : (Iterable) obj) {
                wrapUnknown(arrayBuilder.element(), element);
            }
            return arrayBuilder.endArray();
        }
        return builder.valueObjectAsString(obj);
    }


    private List<Pair<JsonField, Object>> getFields() {
        List<Pair<JsonField, Object>> res = new ArrayList<>();
        Field[] fields = data.getClass().getFields();
        for (Field field : fields) {
            if (Modifier.isPublic(field.getModifiers())) {
                JsonField fieldAnnotation = field.getAnnotation(JsonField.class);
                if (fieldAnnotation != null) {
                    try {
                        res.add(new Pair<>(fieldAnnotation, field.get(data)));
                    } catch (IllegalAccessException e) {
                        log.error("Error accessing annotated field " + field + " of exception " + data, e);
                    }
                }
            }
        }
        return res;
    }

    @Override
    public <C extends ContainerBuilder> JsonObjectBuilder<C> doWriteObject(JsonObjectBuilder<C> builder) {
        String exception = data.getClass().getName();
        builder = builder
                .keyValue("code", data.getProblem().toString())
                .keyValueBoolean("isInternal", false)
                .keyValue("message", data.getMessage())
                .keyValue("exception", exception);

        JsonObjectBuilder<JsonObjectBuilder<C>> paramsBuilder = builder.key("params").valueObject();
        for (Pair<JsonField, Object> entry : getFields()) {
            paramsBuilder = wrapUnknown(paramsBuilder.key(entry.first.name(), entry.first.description()), entry.second);
        }
        return paramsBuilder.endObject();
    }

    @Override
    public void toXml(StringBuilder storage) {
        new CodeErrorInfo(data.getProblem(), data.getProblem().getId(), data.getMessage(), null, data, data.getExtraParams()).toXml(storage);
    }
}
