package ru.yandex.autodoc.wmtools.params.fetch;

import ru.yandex.autodoc.wmtools.errors.CommonUserException;
import org.apache.commons.validator.routines.EmailValidator;
import org.jetbrains.annotations.Contract;
import ru.yandex.wmtools.common.data.info.IPInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.service.ValidatorService;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * @author avhaliullin
 */
public class ParamUtils {

    public static ParamHolder<URL> getUrl(ParamHolder<String> url) {
        URL result = null;
        if (url.getValue() == null) {
            return url.map(null);
        }
        try {
            result = AbstractServantlet.prepareUrl(url.getValue(), true);
        } catch (UserException e) {
            url.errorMalformedURL();
        }
        return url.map(result);
    }

    public static URL getUrlValue(ParamHolder<String> url, int lengthLimit) {
        URL result = getUrlValue(url);
        if (result != null && result.toString().length() > lengthLimit) {
            url.errorValueTooLong(lengthLimit);
        }
        return result;
    }

    public static URL getUrlValue(ParamHolder<String> url) {
        return getUrl(url).getValue();
    }

    public static ParamHolder<IPInfo> getIP(ParamHolder<String> ipPH) {
        return getIP(ipPH, true);
    }

    public static ParamHolder<IPInfo> getIP(ParamHolder<String> ipPH, boolean allowMask) {
        if (ipPH.getValue() == null) {
            return ipPH.map(null);
        }
        IPInfo res = null;
        try {
            res = IPInfo.createFromString(ipPH.getValue());
            if (!allowMask && !res.isSingleAddress()) {
                ipPH.addError(CommonUserException.createMalformedIP(ipPH.getName(), ipPH.getValue()));
            }
        } catch (UserException e) {
            ipPH.addError(CommonUserException.createMalformedIP(ipPH.getName(), ipPH.getValue()));
        }
        return ipPH.map(res);
    }

    public static IPInfo getIPValue(ParamHolder<String> ip) {
        return getIPValue(ip, true);
    }

    public static IPInfo getIPValue(ParamHolder<String> ip, boolean allowMask) {
        return getIP(ip, allowMask).getValue();
    }

    public static List<IPInfo> getIPList(List<ParamHolder<String>> ips) {
        return getIPList(ips, true);
    }

    public static List<IPInfo> getIPList(List<ParamHolder<String>> ips, boolean allowMask) {
        List<IPInfo> infos = new ArrayList<>();

        for (ParamHolder<String> ipPH : ips) {
            infos.add(getIPValue(ipPH, allowMask));
        }

        return infos;
    }

    public static String getEmail(ParamHolder<String> emailPH) {
        if (!emailPH.hasError() && emailPH.getValue() != null) {
            if (!EmailValidator.getInstance().isValid(emailPH.getValue())) {
                emailPH.addError(CommonUserException.createMalformedEmail(emailPH.getName(), emailPH.getValue()));
            }
        }
        return emailPH.getValue();
    }

    public static String checkEmail(ValidatorService validatorService, ParamHolder<String> emailField, long userId) throws InternalException {
        String email = emailField.getValue();
        if (email != null && !validatorService.isEmailValid(userId, email)) {
            emailField.addError(CommonUserException.createEmailNotVerified(emailField.getName(), email, userId));
        }
        return email;
    }

    @Contract("null, _ -> null; !null, _ -> !null")
    public static <T> List<T> unwrap(List<ParamHolder<T>> holders, boolean skipErrors) {
        if (holders == null) {
            return null;
        }
        List<T> result = new ArrayList<>();
        for (ParamHolder<T> holder : holders) {
            if (!skipErrors || !holder.hasError()) {
                result.add(holder.getValue());
            }
        }
        return result;
    }

    @Contract("null -> null; !null -> !null")
    public static <T> List<T> unwrap(List<ParamHolder<T>> holders) {
        return unwrap(holders, false);
    }

    @Contract("null -> null; !null -> !null")
    public static <T> List<T> unwrap(ParamHolder<List<ParamHolder<T>>> holders) {
        return unwrap(holders, false);
    }

    @Contract("null, _ -> null; !null, _ -> !null")
    public static <T> List<T> unwrap(ParamHolder<List<ParamHolder<T>>> holders, boolean skipErrors) {
        List<ParamHolder<T>> values = holders.getValue();
        if (values == null) {
            return null;
        } else {
            return unwrap(values, skipErrors);
        }
    }

    @Contract("null -> null; !null -> !null")
    public static <T> List<ParamHolder<T>> filterErrors(List<ParamHolder<T>> xs) {
        if (xs == null) {
            return null;
        }
        List<ParamHolder<T>> res = new ArrayList<>();
        for (ParamHolder<T> ph : xs) {
            if (!ph.hasError()) {
                res.add(ph);
            }
        }
        return res;
    }

    /**
     * Конвертирует строки в урлы, репортит об ошибках через ParamHolder.
     * Гарантирует сохранение количества и порядка следования параметров.
     *
     * @param raw
     * @return
     */
    @Contract("null -> null; !null -> !null")
    public static List<ParamHolder<URL>> checkUrls(List<ParamHolder<String>> raw) {
        if (raw == null) {
            return null;
        }
        List<ParamHolder<URL>> res = new ArrayList<>();
        for (ParamHolder<String> rawUrl : raw) {
            res.add(getUrl(rawUrl));
        }
        return res;
    }

}
