package ru.yandex.autodoc.wmtools.params.fetch.json;

import ru.yandex.autodoc.common.out.json.JsonValueWriter;
import ru.yandex.autodoc.common.out.json.builder.JsonValueBuilder;
import ru.yandex.common.util.collections.Cf;
import ru.yandex.autodoc.common.doc.DocUtils;
import ru.yandex.autodoc.common.out.json.builder.ContainerBuilder;
import ru.yandex.autodoc.common.out.json.builder.JsonArrayBuilder;
import ru.yandex.autodoc.wmtools.params.fetch.ParamHolder;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @author avhaliullin
 */
public class JsonArrayView<T, I> extends AbstractJsonValueView<List<ParamHolder<T>>, I> {
    private final Integer minValues;
    private final Integer maxValues;

    protected final JsonValueView<T, I> elementView;


    public JsonArrayView(JsonValueView<T, I> elementView, Integer minValues, Integer maxValues) {
        this.elementView = elementView;
        this.maxValues = maxValues;
        this.minValues = minValues;
    }

    public JsonArrayView(JsonValueView<T, I> elementView) {
        this(elementView, null, null);
    }

    @Override
    public ParamHolder<List<ParamHolder<T>>> fetch(JsonValueContext context, I injected) {
        JsonArrayContext arrayContext = context.asArray();
        List<JsonValueContext> elementContexts = arrayContext.getItems();
        List<ParamHolder<T>> res = new ArrayList<>();
        if (elementContexts != null) {
            for (JsonValueContext elementContext : elementContexts) {
                res.add(elementView.fetch(elementContext, injected));
            }
        }
        ParamHolder<List<ParamHolder<T>>> resultHolder =
                new ParamHolder<>(arrayContext.jsonParamName(), res, arrayContext.queryContext, false, String.valueOf(arrayContext.node));
        if (minValues != null && minValues > res.size()) {
            resultHolder.errorTooFewValues(minValues, res.size());
        }
        if (maxValues != null && maxValues < res.size()) {
            resultHolder.errorTooManyValues(maxValues, res.size());
        }
        return resultHolder;
    }

    @Override
    public List<ParamHolder<T>> fetchValue(JsonValueContext context, I injected) {
        return fetch(context, injected).getValue();
    }

    private String className(Object o, String dflt) {
        String result = DocUtils.getNameForObject(o);
        if (result == null) {
            return dflt;
        } else {
            return result;
        }
    }

    @Override
    public JsonValueWriter getExample(final Set<JsonValueView> usedViews) {
        return new JsonValueWriter() {
            @Override
            public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                JsonArrayBuilder<C> arrayBuilder = builder.valueArray();
                if (elementView instanceof JsonObjectView && usedViews.contains(elementView)) {
                    String typeName = className(elementView, "Types, used recursively, should have names!");
                    arrayBuilder = arrayBuilder.element().valueObjectName(typeName);
                } else {
                    arrayBuilder = arrayBuilder.element().writeValue(elementView.getExample(usedViews));
                }
                return arrayBuilder.endArray();
            }
        };
    }

    @Override
    public List<String> getExtraInfo() {
        List<String> res = new ArrayList<>();
        if (minValues != null) {
            res.add("Min values count: " + minValues);
        }
        if (maxValues != null) {
            res.add("Max values count: " + maxValues);
        }
        return res;
    }

    public static class IndexedJsonArrayView<T, I> extends AbstractJsonValueView<List<ParamHolder<T>>, I> {
        protected final JsonValueView<T, InjectableWithIndex<I>> elementView;

        public IndexedJsonArrayView(JsonValueView<T, InjectableWithIndex<I>> elementView) {
            this.elementView = elementView;
        }

        @Override
        public ParamHolder<List<ParamHolder<T>>> fetch(JsonValueContext context, I injected) {
            JsonArrayContext arrayContext = context.asArray();
            List<JsonValueContext> elementContexts = arrayContext.getItems();
            List<ParamHolder<T>> res = new ArrayList<>();
            if (elementContexts != null) {
                int index = 0;
                for (JsonValueContext elementContext : elementContexts) {
                    res.add(elementView.fetch(elementContext, new InjectableWithIndex<>(index++, injected)));
                }
            }
            return new ParamHolder<>(arrayContext.jsonParamName(), res, arrayContext.queryContext, false, String.valueOf(arrayContext.node));
        }

        @Override
        public List<ParamHolder<T>> fetchValue(JsonValueContext context, I injected) {
            return fetch(context, injected).getValue();
        }

        private String className(Object o, String dflt) {
            String result = DocUtils.getNameForObject(o);
            if (result == null) {
                return dflt;
            } else {
                return result;
            }
        }

        @Override
        public JsonValueWriter getExample(final Set<JsonValueView> usedViews) {
            return new JsonValueWriter() {
                @Override
                public <C extends ContainerBuilder> C writeValue(JsonValueBuilder<C> builder) {
                    JsonArrayBuilder<C> arrayBuilder = builder.valueArray();
                    if (elementView instanceof JsonObjectView && usedViews.contains(elementView)) {
                        String typeName = className(elementView, "Types, used recursively, should have names!");
                        arrayBuilder = arrayBuilder.element().valueObjectName(typeName);
                    } else {
                        arrayBuilder = arrayBuilder.element().writeValue(elementView.getExample(usedViews));
                    }
                    return arrayBuilder.endArray();
                }
            };
        }

        @Override
        public List<String> getExtraInfo() {
            return Cf.list();
        }
    }

}
