package ru.yandex.featurer.service

import java.io.{FileWriter, BufferedWriter, File}
import java.time.ZonedDateTime
import java.util.Date

import com.typesafe.config.{ConfigFactory, Config}
import ru.yandex.featurer.data._
import ru.yandex.featurer.utils.SprayJsonUtils
import spray.json.{JsonParser, DefaultJsonProtocol}

import scala.io.Source

/**
 * @author avhaliullin
 */
trait ApplicationsServiceComponent {
  def applicationsService: ApplicationsService

  class ApplicationsService(conf: ApplicationsService.ApplicationsServiceConf) {

    private object JsProtocol extends DefaultJsonProtocol {
      implicit val zonedDateTimeFormat = SprayJsonUtils.delegate[ZonedDateTime, String](_.toString, ZonedDateTime.parse(_))
      val AppFeatureInfoFormat = jsonFormat2(ApplicationFeatureInfo)
    }

    def getApplicationInfo(appId: AppId): Option[ApplicationInfo] = {
      val confFile = new File(conf.confDir + "/" + appId.id + ".conf")
      if (!confFile.isFile) {
        None
      } else {
        val appConf = ConfigFactory.parseFile(confFile)
        Some(ApplicationInfo(
          appId,
          TeamcityBuildId(appConf.getString("teamcityBuildId")),
          appConf.getInt("port"),
          UbicService(appConf.getString("service.name")),
          appConf.getString("baseContainerName")
        ))
      }
    }

    private def getFeatureInfo(file: File) = {
      val content = Source.fromFile(file).mkString
      val json = JsonParser(content)
      JsProtocol.AppFeatureInfoFormat.read(json)
    }

    def getApplicationFeatureInfo(appId: AppId, featureId: FeatureId): Option[ApplicationFeatureInfo] = {
      val file = new File(conf.contentDir + "/" + appId.id + "/" + featureId.id + ".json")
      if (file.isFile) {
        Some(getFeatureInfo(file))
      } else {
        None
      }
    }

    def deleteFeatureInfo(appId: AppId, featureId: FeatureId) = {
      val file = new File(conf.contentDir + "/" + appId.id + "/" + featureId.id + ".json")
      if (file.isFile) {
        if (!file.delete()) {
          throw new RuntimeException("Failed to delete file " + file)
        }
      }
    }

    def updateFeatureInfo(appId: AppId, appFeatureInfo: ApplicationFeatureInfo) = {
      val dir = new File(conf.contentDir + "/" + appId.id)
      if (!dir.isDirectory) {
        if (!dir.mkdir()) {
          throw new RuntimeException("Failed to create directory " + dir)
        }
      }
      val file = new File(conf.contentDir + "/" + appId.id + "/" + appFeatureInfo.featureId.id + ".json")
      val writer = new BufferedWriter(new FileWriter(file))
      writer.append(JsProtocol.AppFeatureInfoFormat.write(appFeatureInfo).compactPrint)
      writer.close()
    }

    def listApplications: Seq[ApplicationInfo] = {
      val dir = new File(conf.confDir)
      dir.listFiles().collect {
        case file: File if file.getName.endsWith(".conf") =>
          val appId = AppId(file.getName.substring(0, file.getName.length - ".conf".length))
          getApplicationInfo(appId).get
      }
    }

    def listFeatures: Seq[(ApplicationInfo, Seq[ApplicationFeatureInfo])] = {
      val dir = new File(conf.contentDir)
      dir.listFiles().collect {
        case file: File if file.isDirectory =>
          val appId = AppId(file.getName)
          (getApplicationInfo(appId).get, listApplicationFeatures(appId))
      }
    }

    def listApplicationFeatures(appId: AppId): Seq[ApplicationFeatureInfo] = {
      val dir = new File(conf.contentDir + "/" + appId.id)
      if (!dir.isDirectory) {
        Nil
      } else {
        dir.listFiles().collect {
          case file: File if file.getName.endsWith(".json") =>
            getFeatureInfo(file)
        }
      }
    }
  }

  object ApplicationsService {

    case class ApplicationsServiceConf(contentDir: String, confDir: String)

    object ApplicationsServiceConf {
      implicit def apply(config: Config): ApplicationsServiceConf =
        new ApplicationsServiceConf(config.getString("contentDir"), config.getString("confDir"))
    }

  }

}
