package ru.yandex.featurer.service

import java.io.{File, BufferedWriter, FileWriter}

import com.typesafe.config.Config
import org.slf4j.LoggerFactory
import ru.yandex.featurer.data.{AppId, ApplicationInfo, FeatureId}

import scala.sys.process.{Process, ProcessLogger}

/**
 * @author avhaliullin
 */
trait NginxServiceComponent {
  def nginxService: NginxService

  class NginxService(config: NginxService.NginxServiceConfig) {
    private val log = LoggerFactory.getLogger(getClass)
    private val pl = ProcessLogger(log.info(_), log.error(_))

    private def confFileName(appId: AppId, featureId: FeatureId) = "featurer_" + appId.id + "_" + featureId.id + ".conf"
    private def defaultConfFileName(appId: AppId) = "featurer_" + appId.id + ".default.conf"

    def restart(): Unit = {
      Process("sudo", Seq("service", "nginx", "reload")) !! pl
    }

    private def addDefaultConfiguration(appInfo: ApplicationInfo) = {
      val fName = defaultConfFileName(appInfo.appId)
      val w = new BufferedWriter(new FileWriter(config.confDir + "/" + fName))
      w.append(
        s"""
           |server {
           |  listen [::]:${appInfo.port} default_server;
           |  listen ${appInfo.port} default_server;
           |}
         """.stripMargin)
      w.close()
    }

    def addConfiguration(appInfo: ApplicationInfo, featureId: FeatureId, ip: String) = {
      addDefaultConfiguration(appInfo)
      val fName = confFileName(appInfo.appId, featureId)
      val w = new BufferedWriter(new FileWriter(config.confDir + "/" + fName))
      w.append(
        s"""
           |server {
           |  listen [::]:${appInfo.port};
           |  listen ${appInfo.port};
           |  server_name ${featureId.id}.${config.zoneName};
           |  location / {
           |    proxy_pass http://$ip:${appInfo.port}/;
           |  }
           |}
         """.stripMargin)
      w.close()
    }

    def deleteConfiguration(appId: AppId, featureId: FeatureId): Unit = {
      val f = new File(config.confDir + "/" + confFileName(appId, featureId))
      if (f.isFile) {
        if (!f.delete()) {
          throw new RuntimeException("Failed to delete " + f)
        }
      }
    }
  }

  object NginxService {

    case class NginxServiceConfig(zoneName: String, confDir: String)

    object NginxServiceConfig {
      implicit def apply(config: Config): NginxServiceConfig = NginxServiceConfig(config.getString("dnsZoneName"), config.getString("confDir"))
    }

  }

}
