package ru.yandex.featurer.service

import java.util.concurrent.atomic.AtomicBoolean

import com.typesafe.config.Config
import org.slf4j.LoggerFactory
import ru.yandex.featurer.controller.ApplicationFeatureView
import ru.yandex.featurer.data.{ContainerStateEnum, AppId, ApplicationFeatureInfo, FeatureId}
import ru.yandex.featurer.error.UserException

import scala.util.Try
import scala.util.control.NonFatal

/**
 * @author avhaliullin
 */
trait OperationsServiceComponent {
  component: ApplicationsServiceComponent
    with ContainerServiceComponent
    with NginxServiceComponent
    with TeamcityServiceComponent =>

  def operationsService: OperationsService

  class OperationsService(config: OperationsService.OperationsServiceConfig) {
    private val log = LoggerFactory.getLogger(getClass)

    def startAll(): Unit = {
      var nginxQueue = Vector[() => Unit]()
      applicationsService.listFeatures.foreach {
        case (appInfo, features) =>
          features.foreach {
            featureInfo =>
              try {
                val ip = containerService.createAndStart(appInfo, featureInfo.featureId)
                nginxQueue = nginxQueue :+ { () => nginxService.addConfiguration(appInfo, featureInfo.featureId, ip) }
              } catch {
                case NonFatal(e) => log.error(s"Failed to start container ${appInfo.appId}, ${featureInfo.featureId}", e)
              }
          }
      }
      if (nginxQueue.nonEmpty) {
        nginxQueue.foreach(_())
        nginxService.restart()
      }
    }

    private val updateAllInProgress = new AtomicBoolean(false)

    def updateAll(): Unit = {
      if (updateAllInProgress.compareAndSet(false, true)) {
        try {
          applicationsService.listFeatures.foreach {
            case (appInfo, features) =>
              features.foreach {
                featureInfo =>
                  teamcityService.getArtifactInfo(appInfo, featureInfo.featureId).foreach {
                    artifact =>

                      if (featureInfo.artifactDate.isEmpty || featureInfo.artifactDate.get.isBefore(artifact.modifiedDate)) {
                        log.info(s"Found new artifact for application ${appInfo.appId.id} and feature ${featureInfo.featureId.id}")
                        containerService.installArtifact(appInfo, featureInfo.featureId, artifact)
                        applicationsService.updateFeatureInfo(appInfo.appId, ApplicationFeatureInfo(featureInfo.featureId, Some(artifact.modifiedDate)))
                      }
                  }
              }

          }
        } finally {
          updateAllInProgress.set(false)
        }
      } else {
        log.warn("Update all already in progress, ignoring")
      }
    }

    def listApplications = {
      applicationsService.listApplications
    }

    def getApplicationInfo(appId: AppId) = {
      applicationsService.getApplicationInfo(appId).getOrElse(throw UserException.ResourceNotFound("application", appId.id))
    }

    def deleteFeature(appId: AppId, featureId: FeatureId) = {
      val containerRes = Try(containerService.destroyContainer(appId, featureId))
      val appRes = Try(applicationsService.deleteFeatureInfo(appId, featureId))
      val nginxRes = Try(nginxService.deleteConfiguration(appId, featureId))
      containerRes.get
      appRes.get
      nginxRes.get
    }

    def createFeature(appId: AppId, featureId: FeatureId): Unit = {
      applicationsService.updateFeatureInfo(appId, ApplicationFeatureInfo(featureId, None))
      val application = getApplicationInfo(appId)
      val ip = containerService.createAndStart(application, featureId)
      nginxService.addConfiguration(application, featureId, ip)
      nginxService.restart()
    }

    def installApplication(appId: AppId, featureId: FeatureId) = {
      val application = getApplicationInfo(appId)
      val teamcityInfo = teamcityService.getArtifactInfo(application, featureId).getOrElse(
        throw UserException.IllegalState(s"Artifact for app $appId, feature $featureId doesn't exist in teamcity"))
      val ip = containerService.installArtifact(application, featureId, teamcityInfo)
      nginxService.addConfiguration(application, featureId, ip)
      nginxService.restart()
      applicationsService.updateFeatureInfo(appId, ApplicationFeatureInfo(featureId, Some(teamcityInfo.modifiedDate)))
    }

    def getApplicationFeatureInfo(appId: AppId, featureId: FeatureId) = {
      val application = getApplicationInfo(appId)
      val applicationFeatureInfo = applicationsService.getApplicationFeatureInfo(appId, featureId).getOrElse(throw UserException.ResourceNotFound("Feature of " + appId.id, featureId.id))
      val containerOpt = containerService.getContainerInfo(appId, featureId)
      val artifactOpt = teamcityService.getArtifactInfo(application, featureId)
      val sshStringOpt = containerOpt.flatMap {
        container =>
          if (container.state == ContainerStateEnum.RUNNING) {
            Some(s"ssh ${config.hostName} -t -- sudo lxc-attach -n ${container.name}")
          } else {
            None
          }
      }
      ApplicationFeatureView(application, applicationFeatureInfo, containerOpt, artifactOpt, sshStringOpt)
    }

    def listAppFeatures(appId: AppId) = {
      applicationsService.listApplicationFeatures(appId)
    }
  }

  object OperationsService {

    case class OperationsServiceConfig(hostName: String)

    object OperationsServiceConfig {
      implicit def apply(config: Config): OperationsServiceConfig =
        OperationsServiceConfig(config.getString("host.name"))
    }

  }

}
