#!/bin/bash

set -o errexit

PROJECT_ROOT=$(pwd)
PACKAGING_TOOLS="${PROJECT_ROOT}/packaging-tools"
BUILD_CMD="${PACKAGING_TOOLS}/bin/application_qloud.py"
GEN_QLOUD_SCRIPTS_CMD="${PACKAGING_TOOLS}/bin/gen_qloud_scripts.py"
DOCKER_REGISTRY_PREFIX="registry.yandex.net/webmaster"

function main {
    export SVN_SSH="ssh -v -i /home/robot-webmaster/.ssh/id_rsa"
    export _JAVA_OPTIONS="-Djava.net.preferIPv6Addresses=True -Djava.net.preferIPv4Stack=False"

    parse_args "$@"

    ARTIFACT_ASSEMBLY_NAME="${ARTIFACT_NAME}-assembly"
    ARTIFACT_DIR="${PROJECT_ROOT}/${ARTIFACT_NAME}/"

    pull_packaging_tools

    if [ ${DEPLOY_ONLY} == "false" ]; then
        build_docker_image
        upload_docker_image
    else
        generate_qloud_scripts
    fi

    deploy_docker_image
}

function parse_args {
    ARTIFACT_ENV="unstable"
    DEPLOY_ONLY="false"

    while [ "$1" != "" ]; do
        case $1 in
            "--artifact-name")
                shift
                ARTIFACT_NAME=$1
                ;;
            "--artifact-env")
                shift
                ARTIFACT_ENV=$1
                ;;
            "--artifact-tag")
                shift
                ARTIFACT_TAG=$1
                ;;
            "--cit-client-config")
                shift
                CIT_CLIENT_CFG=$1
                ;;
            "--deploy-only")
                DEPLOY_ONLY="true"
                ;;
            *)
                echo "Unknown option: $1"
                exit 2
        esac
        shift
    done

    if [ -z ${ARTIFACT_NAME} ]; then
        echo "Missing required option: --artifact-name"
        exit 2
    fi
}

function pull_packaging_tools {
    echo "Pulling packaging tools..."

    if [ -e "${PACKAGING_TOOLS}" ]; then
        rm -Rf "${PACKAGING_TOOLS}"
    fi
    svn checkout svn+ssh://robot-webmaster@arcadia.yandex.ru/arc/trunk/arcadia/wmconsole/backend/packaging-tools
}

function generate_qloud_scripts {
    echo "Generate Qloud scripts..."
    cd ${PROJECT_ROOT} && ${GEN_QLOUD_SCRIPTS_CMD} ${ARTIFACT_ASSEMBLY_NAME} || exit 3
}

function build_docker_image {
    echo "Building docker image..."
    cd ${PROJECT_ROOT} && ${BUILD_CMD} ${ARTIFACT_ASSEMBLY_NAME} || exit 3
}

function purge_old_docker_images {
    DOCKER_IMAGES_TO_REMOVE=$(docker images --filter=label="${ARTIFACT_NAME}.version" --filter=before="${ARTIFACT_DOCKER_ID}" -q)

    if [[ -n "${DOCKER_IMAGES_TO_REMOVE}" ]]; then
        echo "Remove old images: ${DOCKER_IMAGES_TO_REMOVE}"
        docker rmi ${DOCKER_IMAGES_TO_REMOVE}
    fi
}

function upload_docker_image {
    pushd ${ARTIFACT_ASSEMBLY_NAME}
    BOOKMARK=`hg identify -B | grep -o -e 'WMC-[0-9]\+' || :`
    if [ -z ${ARTIFACT_TAG} ]; then
        if [ -z ${BOOKMARK} ]; then
            ARTIFACT_TAG=$(dpkg-parsechangelog | fgrep Version | cut -d ' ' -f 2)
        else
            ARTIFACT_TAG=${BOOKMARK}
        fi
    fi

    ARTIFACT_DOCKER_ID=$(cat target/assembly/docker-image-id.txt)
    popd

    ARTIFACT_DOCKER_TAG="${DOCKER_REGISTRY_PREFIX}/${ARTIFACT_NAME}:${ARTIFACT_TAG}"

    echo "Tag and upload docker artifact: ${ARTIFACT_NAME} ${ARTIFACT_DOCKER_TAG}"
    docker tag ${ARTIFACT_DOCKER_ID} ${ARTIFACT_DOCKER_TAG}
    docker push ${ARTIFACT_DOCKER_TAG}

    purge_old_docker_images
}

function deploy_docker_image {
    if [[ -n "${CIT_CLIENT_CFG}" ]]; then
        echo "Deploying to Qloud..."

        printf -v CUSTOM_CONTEXT '{"artifactEnv":"%s","artifactTag":"%s"}' ${ARTIFACT_ENV} ${ARTIFACT_TAG}

        max_retry=20
        counter=0

        until [ $counter -ge $max_retry ]
        do
           citclient --cfg ${CIT_CLIENT_CFG} --custom-context ${CUSTOM_CONTEXT} --qloud-token "${QLOUD_TOKEN}" qloud.deploy && break
           counter=$[$counter+1]
           sleep 3
           echo "Trying again. Try #$counter"
        done
    fi
}

main "$@"
