package ru.yandex.webmaster3.api.addurl.action;

import com.datastax.driver.core.utils.UUIDs;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.addurl.RecrawlSendingStatus;
import ru.yandex.webmaster3.core.addurl.RecrawlState;
import ru.yandex.webmaster3.core.addurl.UrlForRecrawl;
import ru.yandex.webmaster3.core.addurl.UrlRecrawlEventLog;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.DailyQuotaUtil;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.UrlRecrawlTaskData;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.addurl.AddUrlRequestsService;
import ru.yandex.webmaster3.storage.spam.DeepSpamHostFilter;
import ru.yandex.webmaster3.storage.spam.FastSpamHostFilter;

import java.util.UUID;

/**
 * @author avhaliullin
 */
@Description("Добавить URL в переобход")
@Category("addurl")
@Component
public class AddRecrawlUrlAction extends AbstractApiAction<AddRecrawlUrlRequest, AddRecrawlUrlResponse> {
    private final AddUrlRequestsService addUrlRequestsService;
    private final DeepSpamHostFilter deepSpamHostFilter;
    private final FastSpamHostFilter fastSpamHostFilter;
    private final WorkerClient workerClient;
    private final AbtService abtService;

    @Autowired
    public AddRecrawlUrlAction(
            AddUrlRequestsService addUrlRequestsService,
            DeepSpamHostFilter deepSpamHostFilter,
            FastSpamHostFilter fastSpamHostFilter,
            WorkerClient workerClient,
            AbtService abtService) {
        super(Permission.COMMON);

        this.addUrlRequestsService = addUrlRequestsService;
        this.deepSpamHostFilter = deepSpamHostFilter;
        this.fastSpamHostFilter = fastSpamHostFilter;
        this.workerClient = workerClient;
        this.abtService = abtService;
    }

    @Override
    public AddRecrawlUrlResponse process(AddRecrawlUrlRequest request) {
        DateTime now = DateTime.now();
        WebmasterHostId hostId = request.getHostId();
        String fullUrl = request.getEntity().getUrl().toExternalForm();

        DailyQuotaUtil.QuotaUsage quota = addUrlRequestsService.getQuotaUsage(hostId, now);
        if (quota.getQuotaRemain() <= 0) {
            LocalDate today = now.toLocalDate();
            DateTime exceededUntil = today.plusDays(1).toDateTimeAtStartOfDay();
            return new AddRecrawlUrlResponse.QuotaExceededResponse(quota.getTodayQuota(), exceededUntil);
        }

        String relativeUrl = addUrlRequestsService.toRelativeUrlWithVerification(hostId, fullUrl);
        if (relativeUrl == null) {
            return new AddRecrawlUrlResponse.InvalidUrl();
        }

        if (addUrlRequestsService.requestExists(hostId, relativeUrl)) {
            return new AddRecrawlUrlResponse.UrlAlreadyAdded();
        }

        UUID urlId = UUIDs.timeBased();
        UrlForRecrawl urlForRecrawl = new UrlForRecrawl(hostId, urlId, relativeUrl, now, now, RecrawlState.NEW);

        boolean spam = fastSpamHostFilter.checkHost(hostId) || deepSpamHostFilter.checkHost(hostId);
        RecrawlSendingStatus status;
        if (spam) {
            // сразу пометим, как в процессе, чтобы не портить мониторинг
            UrlForRecrawl spamUrl = urlForRecrawl.changeState(RecrawlState.IN_PROGRESS, now);
            addUrlRequestsService.add(spamUrl, request.getBalancerRequestId().toString());
            addUrlRequestsService.increaseQuotaUsage(spamUrl);
            status = RecrawlSendingStatus.SPAM;
        } else {
            addUrlRequestsService.add(urlForRecrawl, request.getBalancerRequestId().toString());
            addUrlRequestsService.increaseQuotaUsage(urlForRecrawl);
            status = RecrawlSendingStatus.SENT;

            // И тестинг и прод пишут в один и тот же топик, разница лишь в том,
            // какие хосты попадают под эксперимент.
            if (abtService.isInExperiment(hostId, Experiment.URL_RECRAWL_SAMOVAR)) {
                workerClient.enqueueTask(UrlRecrawlTaskData.fromUrlForRecrawl(urlForRecrawl));
            }
        }

        addUrlRequestsService.saveLogs(new UrlRecrawlEventLog(urlId, fullUrl,
                now, status, "", request.getUserId()));

        return new AddRecrawlUrlResponse.NormalResponse(request.getUserId(), hostId, urlId, quota.getQuotaRemain());
    }
}
