package ru.yandex.webmaster3.api.addurl.action;

import org.joda.time.DateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.addurl.RecrawlQueueItemLocator;
import ru.yandex.webmaster3.api.http.rest.response.ApiPostResponse;
import ru.yandex.webmaster3.api.http.rest.response.errors.ApiErrorCode;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithLocation;
import ru.yandex.webmaster3.core.data.WebmasterHostId;

import java.util.UUID;

/**
 * @author avhaliullin
 */
public interface AddRecrawlUrlResponse extends ApiPostResponse {
    class NormalResponse extends ApiPostResponse.Accepted202 implements AddRecrawlUrlResponse, ResponseWithLocation<RecrawlQueueItemLocator> {
        private final long userId;
        private final WebmasterHostId hostId;
        private final UUID taskId;
        private final int quotaRemainder;

        public NormalResponse(long userId, WebmasterHostId hostId, UUID taskId, int quotaRemainder) {
            this.userId = userId;
            this.hostId = hostId;
            this.taskId = taskId;
            this.quotaRemainder = quotaRemainder;
        }

        @Override
        public RecrawlQueueItemLocator getLocation() {
            return new RecrawlQueueItemLocator(userId, hostId, taskId);
        }

        @Description("id задачи на переобход")
        public UUID getTaskId() {
            return taskId;
        }

        @Description("Остаток суточной квоты на переобход")
        public int getQuotaRemainder() {
            return quotaRemainder;
        }
    }

    enum Errors implements ApiErrorCode {
        QUOTA_EXCEEDED,
        URL_ALREADY_ADDED,
        INVALID_URL,
    }

    @Description("Суточная квота запросов исчерпана")
    class QuotaExceededResponse extends TooManyRequests429<Errors> implements AddRecrawlUrlResponse {
        private final int dailyQuota;
        private final DateTime exceededUntil;

        public QuotaExceededResponse(int dailyQuota, DateTime exceededUntil) {
            super(Errors.QUOTA_EXCEEDED, "Daily quota exceeded");
            this.dailyQuota = dailyQuota;
            this.exceededUntil = exceededUntil;
        }

        @Description("разрешенное количество запросов в сутки")
        public int getDailyQuota() {
            return dailyQuota;
        }

        @Description("время истечения суток (когда снова можно будет задавать запросы)")
        public DateTime getExceededUntil() {
            return exceededUntil;
        }
    }

    @Description("URL уже в переобходе")
    class UrlAlreadyAdded extends Conflict409<Errors> implements AddRecrawlUrlResponse {
        public UrlAlreadyAdded() {
            super(Errors.URL_ALREADY_ADDED, "Url already added to recrawl");
        }
    }

    @Description("Неправильный URL")
    class InvalidUrl extends BadRequest400<Errors> implements AddRecrawlUrlResponse {
        public InvalidUrl() {
            super(Errors.INVALID_URL, "Invalid Url");
        }
    }
}
