package ru.yandex.webmaster3.api.diagnostics.action;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.diagnostics.data.ApiProblemInfo;
import ru.yandex.webmaster3.api.diagnostics.data.ApiSiteProblemTypeEnum;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;

/**
 * @author avhaliullin
 */
@Description("Посмотреть состояние проблем для сайта")
@Category("diagnostics")
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@ActionPermission(Permission.COMMON)
public class GetHostDiagnosticsAction extends AbstractApiAction<GetHostDiagnosticsRequest, GetHostDiagnosticsResponse> {

    private final MirrorService2 mirrorService2;
    private final SiteProblemsService siteProblemsService;

    @Override
    public GetHostDiagnosticsResponse process(GetHostDiagnosticsRequest request) {
        WebmasterHostId hostId = request.getHostId();
        List<AbstractProblemInfo> problemInfos = siteProblemsService.listProblemsForHost(hostId, null);
        Map<SiteProblemTypeEnum, AbstractProblemInfo> problemInfoMap = problemInfos.stream()
                .collect(Collectors.toMap(AbstractProblemInfo::getProblemType, Function.identity()));

        // NOTE: В API, в отличие от вьювера, мы не фильтруем NOT_APPLICABLE проблемы
        Map<ApiSiteProblemTypeEnum, ApiProblemInfo> responseMap = new HashMap<>();
        boolean isMainMirror = mirrorService2.isMainMirror(request.getHostId());
        for (SiteProblemTypeEnum problemType : SiteProblemTypeEnum.ENABLED_PROBLEMS) {
            AbstractProblemInfo problemInfo = problemInfoMap.get(problemType);
            if (problemInfo == null) {
                problemInfo = new AbstractProblemInfo(hostId, problemType);
            }

            ApiSiteProblemTypeEnum problemTypeEnum = ApiSiteProblemTypeEnum.fromCoreType(problemInfo.getProblemType());
            if (problemTypeEnum == null) {
                // не должно случаться, проверяется тестом ApiSiteProblemTypeEnumTest
                continue;
            }

            responseMap.put(problemTypeEnum, ApiProblemInfo.fromCoreProblemInfo(problemInfo, isMainMirror));
        }

        return new GetHostDiagnosticsResponse(responseMap);
    }
}
