package ru.yandex.webmaster3.api.feeds.action;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncOperationRequestInfo;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoRequestState;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoResultType;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.models.ProcessingAddFeedInputData;
import ru.yandex.webmaster3.storage.robotstxt.model.AnalyzeRobotsTxtResponseType;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor
@Description("Получить список фидов для домена")
@Category("feeds")
@Component
@ActionPermission(Permission.COMMON)
public class FeedsAddInfoAction extends AbstractApiAction<FeedsAddInfoRequest, FeedsAddInfoResponse> {
    private static final Duration ANSWER_WAITING_TIMEOUT = Duration.standardSeconds(70);

    private final AsyncRequestTaskService asyncRequestTaskService;

    @Override
    public FeedsAddInfoResponse process(FeedsAddInfoRequest request) {
        AsyncOperationRequestInfo responseInfo = asyncRequestTaskService.getResponseInfo(request.getEntity().getRequestId());
        if (responseInfo == null) {
            return new FeedsAddInfoResponse.RequestNotFoundResponse();
        }

        ProcessingAddFeedInputData data = JsonMapping.readValue(responseInfo.getRequestData(),
                ProcessingAddFeedInputData.class);
        if (!data.getHId().equals(request.getHostId())) {
            return new FeedsAddInfoResponse.RequestNotFoundResponse();
        }

        if (responseInfo.getState() == HttpAsyncRequestInfoRequestState.NEW &&
                responseInfo.getLastUpdate().plus(ANSWER_WAITING_TIMEOUT).isBefore(DateTime.now())) {
            log.error("Worker failed to finish request in {}", ANSWER_WAITING_TIMEOUT);
            return new FeedsAddInfoResponse.TimedOutResponse();
        }

        if (responseInfo.getState() != HttpAsyncRequestInfoRequestState.TASK_FINISHED) {
            return new FeedsAddInfoResponse.NormalResponse(FeedsAddInfoResponse.SuccessStatus.IN_PROGRESS);
        }
        if (responseInfo.getResult() != HttpAsyncRequestInfoResultType.OK) {
            if (responseInfo.getResult() == HttpAsyncRequestInfoResultType.INTERNAL_ERROR) {
                throw new WebmasterException("InternalError", new WebmasterErrorResponse.AsyncZoraErrorResponse(getClass()));
            }
        }
        if (responseInfo.getResponseObjectTypeId() == AnalyzeRobotsTxtResponseType.NORMAL.getValue()) {
            FeedsService.ZoraCheckResult result = convertData(responseInfo, FeedsService.ZoraCheckResult.class);
            return switch (result) {
                case OK -> new FeedsAddInfoResponse.NormalResponse(FeedsAddInfoResponse.SuccessStatus.OK);
                case TIMED_OUT -> new FeedsAddInfoResponse.TimedOutResponse();
                case BAD_CODE -> new FeedsAddInfoResponse.BadHttpCodeResponse();
                case BAD_MIME_TYPE -> new FeedsAddInfoResponse.BadMimeTypeResponse();
                case BANNED -> new FeedsAddInfoResponse.BannedFeedResponse();
            };
        }
        throw new WebmasterException("InternalError", new WebmasterErrorResponse.AsyncAddFeedErrorResponse(getClass()));
    }

    protected <WorkClass> WorkClass convertData(AsyncOperationRequestInfo info, Class<WorkClass> clazz) {
        WorkClass result = null;
        try {
            result = JsonMapping.readValue(info.getResponse(), clazz);
        } catch (Exception e) {
            log.error("Problem with parsing response.", e);
        }
        return result;
    }
}
