package ru.yandex.webmaster3.api.feeds.action;

import java.net.URL;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedServiceType;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.worker.task.feeds.ProcessingAddFeedTaskData;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.async.model.AsyncTaskType;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.models.ProcessingAddFeedInputData;

/**
 * @author avhaliullin
 */
@RequiredArgsConstructor
@Description("Получить список фидов для домена")
@Category("feeds")
@Component
@ActionPermission(Permission.COMMON)
public class FeedsAddStartAction extends AbstractApiAction<FeedsAddStartRequest, FeedsAddStartResponse> {
    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsService feedsService;
    private final AsyncRequestTaskService asyncRequestTaskService;

    @Override
    public FeedsAddStartResponse process(FeedsAddStartRequest request) {
        var entity = request.getEntity().getFeed();
        String domain = WwwUtil.cutWWWAndM(request.getHostId());

        long feedsCount = feedsNative2YDao.count(domain);
        if (feedsCount >= FeedsService.LIMIT_FEEDS_ON_DOMAIN) {
            return new FeedsAddStartResponse.FeedsLimitExceededResponse(FeedsService.LIMIT_FEEDS_ON_DOMAIN);
        }
        List<Integer> regionsId = entity.getRegionIds();
        if (CollectionUtils.isEmpty(regionsId)) {
            regionsId = List.of(FeedsService.RUSSIA);
        }
        for (Integer regionId : regionsId) {
            if (!feedsService.isRussiaRegion(regionId)) {
                return new FeedsAddStartResponse.WrongRegionResponse(regionId);
            }
        }
        URL url = FeedsService.canonizeUrl(entity.getUrl());
        if (url == null) {
            return new FeedsAddStartResponse.IncorrectUrlResponse();
        }
        if (!url.getProtocol().equals("https")) {
            return new FeedsAddStartResponse.FeedsProtocolUnsupportedResponse();
        }

        var urlWithUserInfo = feedsService.parseUrl(entity.getUrl());
        String urlStr = urlWithUserInfo.url();

        NativeFeedType type = entity.getType().getTypeWm();

        if (feedsService.containsFeed(urlStr, type)) {
            return new FeedsAddStartResponse.FeedAlreadyAddedResponse();
        }

        WebmasterHostId hostId = request.getHostId();
        long userId = request.getUserId();
        String login = urlWithUserInfo.login();
        String password = urlWithUserInfo.password();
        UUID taskId = UUIDs.timeBased();


        ProcessingAddFeedInputData processingAddFeedInputData = new ProcessingAddFeedInputData(hostId, urlStr,
                regionsId, userId, type, Set.of(FeedServiceType.SEARCH), login,
                password);

        ProcessingAddFeedTaskData taskData = new ProcessingAddFeedTaskData(taskId, hostId, urlStr,
                regionsId, userId, type, Set.of(FeedServiceType.SEARCH), login,
                password);

        AsyncRequestTaskService.CreateRequestResult requestResult =
                asyncRequestTaskService.createRequest(AsyncTaskType.ADD_FEED, 1, processingAddFeedInputData,
                        taskData);

        return new FeedsAddStartResponse.NormalResponse(requestResult.getRequestId());
    }
}
