package ru.yandex.webmaster3.api.feeds.action;

import java.net.URL;
import java.util.List;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.feeds.info.FeedInfo;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedServiceType;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.feeds.mbi.ChangeRevokeStatusException;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor
@Description("Получить список фидов для домена")
@Category("feeds")
@Component
@ActionPermission(Permission.COMMON)
public class FeedsBatchAddAction extends AbstractApiAction<FeedsBatchAddRequest, FeedsBatchAddResponse> {
    private final FeedsService feedsService;
    private final FeedsNative2YDao feedsNative2YDao;


    @Override
    public FeedsBatchAddResponse process(FeedsBatchAddRequest request) {
        WebmasterHostId hostId = request.getHostId();
        String domain = WwwUtil.cutWWWAndM(hostId);
        List<FeedInfo> feedInfos = request.getEntity().getFeeds();

        long feedsCount = feedsNative2YDao.count(domain);
        if (feedsCount >= FeedsService.LIMIT_FEEDS_ON_DOMAIN) {
            return new FeedsBatchAddResponse.FeedsLimitExceededResponse(FeedsService.LIMIT_FEEDS_ON_DOMAIN);
        }

        if (feedInfos.size() > 50) {
            return new FeedsBatchAddResponse.BatchLimitExceededResponse(50);
        }

        List<FeedsBatchAddResponse.AddFeedInfo> response = feedInfos.stream()
                .map(info -> {
                    if (CollectionUtils.isEmpty(info.getRegionIds())) {
                        return info.withRegionIds(List.of(FeedsService.RUSSIA));
                    }
                    return info;
                })
                .map(info -> new FeedsBatchAddResponse.AddFeedInfo(info.getUrl().toString(),
                        proccessUrl(domain, info.getUrl(), info.getRegionIds(), info.getType().getTypeWm(),
                                request.getUserId())))
                .toList();

        return new FeedsBatchAddResponse.FeedsLimitExceededResponse.NormalResponse(response);
    }

    private FeedsBatchAddResponse.AddStatus proccessUrl(String domain, String rawUrl,
                                                        List<Integer> regionIds, NativeFeedType type,
                                                        long userId) {

        URL url = FeedsService.canonizeUrl(rawUrl);
        if(url == null) {
            return FeedsBatchAddResponse.AddStatus.INCORRECT_URL;
        }
        if (!url.getProtocol().equals("https")) {
            return FeedsBatchAddResponse.AddStatus.NOT_HTTPS;
        }

        for (Integer regionId : regionIds) {
            if (!feedsService.isRussiaRegion(regionId)) {
                return FeedsBatchAddResponse.AddStatus.WRONG_REGION;
            }
        }

        var urlWithUserInfo = feedsService.parseUrl(url);
        String urlStr = urlWithUserInfo.url();
        String login = urlWithUserInfo.login();
        String password = urlWithUserInfo.password();

        if (feedsService.containsFeed(urlStr, type)) {
            return FeedsBatchAddResponse.AddStatus.FEED_ALREADY_ADDED;
        }
        FeedsService.ZoraCheckResult result = feedsService.checkUrl(urlStr, login, password);
        if (result == FeedsService.ZoraCheckResult.OK) {
            try {
                feedsService.addFeed(domain, urlStr,
                        regionIds, userId, login, password,
                        type, Set.of(FeedServiceType.SEARCH));
                return FeedsBatchAddResponse.AddStatus.OK;
            } catch (ChangeRevokeStatusException e) {
                return FeedsBatchAddResponse.AddStatus.BANNED_FEED;
            }
        }
        return switch (result) {
            case OK -> FeedsBatchAddResponse.AddStatus.OK;
            case BAD_CODE -> FeedsBatchAddResponse.AddStatus.BAD_HTTP_CODE;
            case BAD_MIME_TYPE -> FeedsBatchAddResponse.AddStatus.BAD_MIME_TYPE;
            case TIMED_OUT -> FeedsBatchAddResponse.AddStatus.TIMED_OUT;
            case BANNED -> FeedsBatchAddResponse.AddStatus.BANNED_FEED;
        };
    }

}
