package ru.yandex.webmaster3.api.host.action;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.host.HostLocator;
import ru.yandex.webmaster3.api.http.rest.response.ApiPostResponse;
import ru.yandex.webmaster3.api.http.rest.response.errors.ApiErrorCode;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithLocation;
import ru.yandex.webmaster3.core.data.WebmasterHostId;

/**
 * @author avhaliullin
 */
public interface AddHostResponse extends ApiPostResponse {
    class NormalResponse extends Created201<HostLocator> implements AddHostResponse {
        private final long userId;
        private final WebmasterHostId hostId;

        public NormalResponse(long userId, WebmasterHostId hostId) {
            this.userId = userId;
            this.hostId = hostId;
        }

        @Override
        public HostLocator getLocation() {
            return new HostLocator(userId, hostId);
        }

        @Description("id добавленного хоста")
        public WebmasterHostId getHostId() {
            return hostId;
        }
    }

    enum Errors implements ApiErrorCode {
        HOST_ALREADY_ADDED,
        HOSTS_LIMIT_EXCEEDED,
    }

    @Description("Сайт с таким именем уже добавлен в список сайтов пользователя")
    class HostAlreadyAddedResponse extends Conflict409<Errors> implements AddHostResponse, ResponseWithLocation<HostLocator> {
        private final long userId;
        private final WebmasterHostId hostId;
        private final boolean verified;

        public HostAlreadyAddedResponse(long userId, WebmasterHostId hostId, boolean verified) {
            super(Errors.HOST_ALREADY_ADDED, "Host already added with id \"" + hostId + "\"");
            this.userId = userId;
            this.hostId = hostId;
            this.verified = verified;
        }

        @Description("id сайта, который уже добавлен")
        public WebmasterHostId getHostId() {
            return hostId;
        }

        @Description("Подтвержден ли добавленный хост")
        public boolean isVerified() {
            return verified;
        }

        @Override
        public HostLocator getLocation() {
            return new HostLocator(userId, hostId);
        }
    }

    @Description("Превышен лимит количества сайтов в списке сайтов пользователя (текущий лимит - 1703)")
    class HostsLimitExceededResponse extends Forbidden403<Errors> implements AddHostResponse {
        private final int limit;

        public HostsLimitExceededResponse(int limit) {
            super(Errors.HOSTS_LIMIT_EXCEEDED, "Hosts count limit exceeded - max hosts allowed: " + limit);
            this.limit = limit;
        }

        @Description("Лимит количества добавленых сайтов")
        public int getLimit() {
            return limit;
        }
    }
}
