package ru.yandex.webmaster3.api.host.action;

import java.util.Optional;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.host.data.ApiHostDataStatus;
import ru.yandex.webmaster3.api.host.data.ApiHostInfo;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructureNode;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.sitestructure.SiteStructureService;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;

/**
 * @author avhaliullin
 */
@Description("Информация о сайте")
@Category("host")
@Component
@ActionPermission(Permission.COMMON)
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostInfoAction extends AbstractApiAction<HostInfoRequest, HostInfoResponse> {

    private final DisplayNameService2 displayNameService2;
    private final MirrorService2 mirrorService2;
    private final SiteStructureService siteStructureService;
    private final UserHostsService userHostsService;

    @Override
    public HostInfoResponse process(HostInfoRequest request) {
        WebmasterHostId hostId = request.getHostId();
        WebmasterHostId mainMirrorId = mirrorService2.getMainMirror(hostId);
        if (mainMirrorId != null && mainMirrorId.equals(hostId)) {
            mainMirrorId = null;
        }
        Optional<ApiHostInfo> mainMirror = Optional.empty();
        if (mainMirrorId != null) {
            UserVerifiedHost userVerifiedHost =
                    userHostsService.getVerifiedHost(new WebmasterUser(request.getUserId()), hostId);
            boolean mainMirrorVerified = userVerifiedHost != null;
            mainMirror = Optional.of(new ApiHostInfo(mainMirrorId, mainMirrorVerified));
        }
        Optional<String> hostDisplayName = Optional.empty();
        Optional<ApiHostDataStatus> dataStatusOpt = Optional.empty();
        if (request.getUserAddedHost().isVerified()) {
            ApiHostDataStatus dataStatus;
            NewSiteStructureNode node = siteStructureService.getNode(hostId, NewSiteStructure.ROOT_NODE_ID);
            if (node == null) {
                dataStatus = ApiHostDataStatus.NOT_LOADED;
            } else {
                if (node.getDocCount() > 0) {
                    dataStatus = ApiHostDataStatus.OK;
                } else {
                    dataStatus = ApiHostDataStatus.NOT_INDEXED;
                }
            }
            dataStatusOpt = Optional.of(dataStatus);
            hostDisplayName = Optional.ofNullable(displayNameService2.getDisplayName(hostId, true));
        }
        return new HostInfoResponse(hostId, request.getUserAddedHost().isVerified(), mainMirror,
                dataStatusOpt, hostDisplayName);
    }
}
