package ru.yandex.webmaster3.api.http.auth;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.server.Request;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.api.http.rest.request.ApiRequest;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.OAuthClientInfo;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.RetryUtils;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
@Component("apiRequestAuthorizer")
public class OAuthAuthorizer implements ApiRequestAuthorizer<OAuthAuthorizerError> {
    private static final Logger log = LoggerFactory.getLogger(OAuthAuthorizer.class);

    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String OAUTH_PREFIX = "OAuth ";

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(3, Duration.standardSeconds(10));

    private final BlackboxUsersService blackboxExternalYandexUsersService;

    public OAuthAuthorizer(BlackboxUsersService blackboxExternalYandexUsersService) {
        this.blackboxExternalYandexUsersService = blackboxExternalYandexUsersService;
    }

    @Override
    public OAuthAuthorizerError authorize(Permission requiredPermission, Request baseRequest, ApiRequest actionRequest) {
        String authHeader = baseRequest.getHeader(AUTHORIZATION_HEADER);
        if (StringUtils.isEmpty(authHeader)) {
            return new OAuthAuthorizerError.InvalidToken("Authorization header is missing");
        }
        if (!authHeader.startsWith(OAUTH_PREFIX)) {
            return new OAuthAuthorizerError.InvalidToken("Unrecognized authorization header value: expected \"OAuth <token>\"");
        }
        String token = authHeader.substring(OAUTH_PREFIX.length(), authHeader.length());
        OAuthClientInfo clientInfo;
        try {
            clientInfo = RetryUtils.query(RETRY_POLICY, () -> blackboxExternalYandexUsersService.checkOAuthToken(token));
        } catch (InterruptedException e) {
            log.warn("Exception during checking token", e);
            throw new WebmasterException("Failed to query blackbox",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
        }
        if (clientInfo == null) {
            return new OAuthAuthorizerError.InvalidToken("Used OAuth token is invalid or expired");
        }
        if (log.isInfoEnabled()) {
            log.info("Authenticated client {} user {} with scopes [{}]",
                    clientInfo.getClientId(),
                    clientInfo.getUid(),
                    clientInfo.getScope()
                            .stream()
                            .map(scope -> scope.getService() + ":" + scope.getScope())
                            .collect(Collectors.joining(","))
            );
        }
        Set<Permission> permissions = EnumSet.noneOf(Permission.class);
        for (OAuthClientInfo.Scope scope : clientInfo.getScope()) {
            OAuthScope webmasterScope = OAuthScope.getScope(scope.getService(), scope.getScope());
            if (webmasterScope != null) {
                permissions.addAll(webmasterScope.getPermissions());
            }
        }
        if (!permissions.contains(requiredPermission)) {
            return new OAuthAuthorizerError.AccessForbidden(requiredPermission, new ArrayList<>(permissions));
        }

        // WMC-7648
        if (clientInfo.getUid() == 785316385) {
            return new OAuthAuthorizerError.AccessForbidden();
        }

        actionRequest.setAuthorizedUserId(clientInfo.getUid());
        return null;
    }
}
