package ru.yandex.webmaster3.api.http.rest.autodoc;

import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.introspect.AnnotatedMember;
import com.fasterxml.jackson.databind.introspect.BeanPropertyDefinition;
import com.google.common.collect.ImmutableSet;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.autodoc.common.doc.types.AnyObjectModel;
import ru.yandex.autodoc.common.doc.types.CollectionType;
import ru.yandex.autodoc.common.doc.types.FieldDescription;
import ru.yandex.autodoc.common.doc.types.MapType;
import ru.yandex.autodoc.common.doc.types.ObjectModel;
import ru.yandex.autodoc.common.doc.types.PolyObjectModel;
import ru.yandex.autodoc.common.doc.types.ValueType;
import ru.yandex.webmaster3.api.http.rest.jackson.WebmasterApiJacksonModule;
import ru.yandex.webmaster3.api.http.rest.jackson.xml.PluralNameTransformer;
import ru.yandex.webmaster3.api.http.rest.response.errors.ApiErrorCode;
import ru.yandex.webmaster3.api.http.rest.types.WebmasterApiTypes;
import ru.yandex.webmaster3.core.http.TypeDescriptionResolver;
import ru.yandex.webmaster3.core.http.autodoc.EnumComparators;
import ru.yandex.webmaster3.core.http.autodoc.FullTypeInfo;
import ru.yandex.webmaster3.core.util.json.polymorphic.Discriminator;
import ru.yandex.webmaster3.core.util.json.polymorphic.Polymorphic;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @author avhaliullin
 */
public class ObjectModelResolver {
    private static final String OPTIONAL_COMMENT = "optional";
    //TODO: мне не нравится то, что Data приходится дублировать тут и в (De)SerializationConfig для jackson.
    // Но сюда мы передаем JSON (а не XML) конфиг, поэтому root name непосредственно из конфига не забрать
    // А есть JSON-конфигу просетать root name - то он реально будет дозаворачивать корневые объекты
    private static final String ROOT_OBJECT_NAME = "Data";

    private final WebmasterApiJacksonModule webmasterApiJacksonModule;
    private final ObjectMapper objectMapper;

    public ObjectModelResolver(WebmasterApiJacksonModule webmasterApiJacksonModule, ObjectMapper objectMapper) {
        this.webmasterApiJacksonModule = webmasterApiJacksonModule;
        this.objectMapper = objectMapper;
    }

    public AnyObjectModel resolveRequestObjectModel(FullTypeInfo type) {
        return resolveObject(Mode.REQUEST, ImmutableSet.of(type), type, ROOT_OBJECT_NAME);
    }

    public AnyObjectModel resolveResponseObjectModel(FullTypeInfo type) {
        return resolveObject(Mode.RESPONSE, ImmutableSet.of(type), type, ROOT_OBJECT_NAME);
    }

    public ObjectModel resolveSimpleModel(FullTypeInfo type) {
        return resolveSimpleObject(Mode.RESPONSE, ImmutableSet.of(type), type, ROOT_OBJECT_NAME);
    }

    /**
     * @param usedTypes типы, использованные на пути от корня, НЕ включая текущий
     */
    public ValueType resolveType(Mode mode, ImmutableSet<FullTypeInfo> usedTypes, FullTypeInfo type) {
        type = forceConcreteType(type, mode);
        if (usedTypes.contains(type)) {
            return new ObjectModel(type.getClazz().getSimpleName(), Collections.emptyList(), null);
        }
        usedTypes = addToSet(usedTypes, type);
        {
            ValueType primitive = getPrimitiveResolver(mode).describeType(type.getClazz());
            if (primitive != null) {
                return primitive;
            }
            if (type.getClazz().isEnum()) {
                return WebmasterApiTypes.enumType((Class<Enum<?>>) type.getClazz());
            }
        }
        if (Iterable.class.isAssignableFrom(type.getClazz())) {
            FullTypeInfo itemType = type.ancestorFullType(Iterable.class).getGenericsList().get(0);
            ValueType resolvedItem = resolveType(mode, usedTypes, itemType);
            return new CollectionType(resolvedItem);
        }

        if (Map.class.isAssignableFrom(type.getClazz())) {
            FullTypeInfo parameterizedMapType = type.ancestorFullType(Map.class);
            FullTypeInfo keyType = parameterizedMapType.getGenericsList().get(0);
            FullTypeInfo valueType = parameterizedMapType.getGenericsList().get(1);

            ValueType keyValueType = resolveType(mode, usedTypes, keyType);
            ValueType valueValueType = resolveType(mode, usedTypes, valueType);
            return new MapType(keyValueType, valueValueType);
        }
        return resolveObject(mode, usedTypes, type, null);
    }

    /**
     * @param usedTypes типы, использованные на пути от корня, ВКЛЮЧАЯ текущий
     */
    private ObjectModel resolveSimpleObject(Mode mode, ImmutableSet<FullTypeInfo> usedTypes, FullTypeInfo type, String objectName) {
        BeanDescription beanDescription;
        switch (mode) {
            case REQUEST:
                beanDescription = objectMapper.getDeserializationConfig().introspect(toJacksonType(type));
                break;
            case RESPONSE:
                beanDescription = objectMapper.getSerializationConfig().introspect(toJacksonType(type));
                break;
            default:
                throw new RuntimeException("Unknown resolve mode " + mode);
        }

        List<FieldDescription> fields = new ArrayList<>();
        for (BeanPropertyDefinition property : beanDescription.findProperties()) {
            AnnotatedMember member = property.getPrimaryMember();
            if (member != null) {
                FullTypeInfo propertyType = type.ancestorFullType(member.getDeclaringClass())
                        .memberFullType(member.getGenericType());
                propertyType = forceConcreteType(propertyType, mode);

                if (propertyType.isConcrete()) {
                    boolean isDiscriminatorField = ApiErrorCode.class.isAssignableFrom(propertyType.getClazz());
                    isDiscriminatorField |= Discriminator.class.isAssignableFrom(propertyType.getClazz()) &&
                            Polymorphic.class.isAssignableFrom(type.getClazz());
                    if (isDiscriminatorField) {
                        continue;
                    }
                }
                String name = property.getName();
                String itemName = name;
                if (name.endsWith("s")) {
                    itemName = PluralNameTransformer.INSTANCE.transform(name);
                }
                String description = property.getMetadata().getDescription();

                boolean required = true;
                if (Optional.class.isAssignableFrom(propertyType.getClazz())) {
                    propertyType = propertyType.getGenericsList().get(0);
                    required = false;
                }
                ValueType propertyValueType = resolveType(mode, usedTypes, propertyType);
                if (!required) {
                    if (description == null) {
                        description = OPTIONAL_COMMENT;
                    } else {
                        description = OPTIONAL_COMMENT + ". " + description;
                    }
                }
                fields.add(new FieldDescription(required, propertyValueType, description, name, itemName));
            }
        }
        Description objectDescriptionAnnotation = type.getClazz().getAnnotation(Description.class);
        String objectDescription = objectDescriptionAnnotation == null ? null : objectDescriptionAnnotation.value();

        if (objectName == null) {
            objectName = type.getClazz().getSimpleName();
        }
        return new ObjectModel(objectName, fields, objectDescription);
    }

    /**
     * @param usedTypes типы, использованные на пути от корня, ВКЛЮЧАЯ текущий
     */
    private AnyObjectModel resolveObject(Mode mode, ImmutableSet<FullTypeInfo> usedTypes, FullTypeInfo type, String objectName) {
        int classModifiers = type.getClazz().getModifiers();
        if (Modifier.isAbstract(classModifiers) || Modifier.isInterface(classModifiers)) {
            if (Polymorphic.class.isAssignableFrom(type.getClazz())) {
                objectName = objectName == null ? type.getClazz().getSimpleName() : objectName;
                FullTypeInfo polyParameterizedType = type.ancestorFullType(Polymorphic.class);
                FullTypeInfo discriminatorType = polyParameterizedType.getGenericsList().get(0);
                Class<Enum> enumClass = (Class<Enum>) discriminatorType.getClazz();
                Comparator enumComparator = EnumComparators.createComparator(enumClass);
                List<Enum> enumValues = Arrays.asList(enumClass.getEnumConstants());
                Collections.sort(enumValues, enumComparator);
                List<PolyObjectModel.Case> cases = new ArrayList<>();
                for (Enum discriminator : enumValues) {
                    Class<?> caseClass = ((Discriminator) discriminator).getDataClass();
                    if (type.getClazz().isAssignableFrom(caseClass)) {
                        FullTypeInfo caseType = FullTypeInfo.createSimple(caseClass);
                        ObjectModel caseModel = resolveSimpleObject(mode, addToSet(usedTypes, caseType), caseType, objectName);
                        cases.add(new PolyObjectModel.Case(discriminator.name(), caseModel));
                    }
                }
                return new PolyObjectModel("type", objectName, cases);
            }
        }
        return resolveSimpleObject(mode, usedTypes, type, objectName);
    }

    private FullTypeInfo forceConcreteType(FullTypeInfo type, Mode mode) {
        switch (mode) {
            case REQUEST:
                return type.thisOrLower();
            case RESPONSE:
                return type.thisOrUpper();
            default:
                throw new RuntimeException("Unknown mode " + mode);
        }
    }

    private <T> ImmutableSet<T> addToSet(ImmutableSet<T> set, T value) {
        return ImmutableSet.<T>builder().addAll(set).add(value).build();
    }

    private TypeDescriptionResolver getPrimitiveResolver(Mode mode) {
        switch (mode) {
            case REQUEST:
                return webmasterApiJacksonModule.getDeserializingTypeResolver();
            case RESPONSE:
                return webmasterApiJacksonModule.getSerializingTypeResolver();
            default:
                throw new RuntimeException("Unknown mode " + mode);
        }
    }

    private JavaType toJacksonType(FullTypeInfo typeInfo) {
        if (typeInfo.isConcrete()) {
            JavaType[] generics = new JavaType[typeInfo.getGenericsList().size()];
            for (int i = 0; i < typeInfo.getGenericsList().size(); i++) {
                generics[i] = toJacksonType(typeInfo.getGenericsList().get(i));
            }
            return objectMapper.getTypeFactory().constructParametrizedType(typeInfo.getClazz(), typeInfo.getClazz(), generics);
        } else {
            if (typeInfo.getUpperBounds().isEmpty()) {
                return objectMapper.constructType(Object.class);
            } else {
                return toJacksonType(typeInfo.getUpperBounds().get(0));
            }
        }
    }

    public enum Mode {
        REQUEST,
        RESPONSE,
    }
}
