package ru.yandex.webmaster3.api.http.rest.jackson;

import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.deser.std.UUIDDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.fasterxml.jackson.databind.ser.std.UUIDSerializer;
import com.fasterxml.jackson.datatype.joda.cfg.FormatConfig;
import com.fasterxml.jackson.datatype.joda.cfg.JacksonJodaDateFormat;
import com.fasterxml.jackson.datatype.joda.deser.DateTimeDeserializer;
import com.fasterxml.jackson.datatype.joda.deser.InstantDeserializer;
import com.fasterxml.jackson.datatype.joda.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.joda.ser.DateTimeSerializer;
import com.fasterxml.jackson.datatype.joda.ser.InstantSerializer;
import com.fasterxml.jackson.datatype.joda.ser.LocalDateSerializer;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import ru.yandex.autodoc.common.doc.types.ValueType;
import ru.yandex.webmaster3.api.host.data.ASCIIHostUrl;
import ru.yandex.webmaster3.api.host.data.UnicodeHostUrl;
import ru.yandex.webmaster3.api.http.rest.jackson.deser.ApiURLDeserializer;
import ru.yandex.webmaster3.api.http.rest.jackson.ser.ApiSitemapIdSerializer;
import ru.yandex.webmaster3.api.http.rest.types.WebmasterApiTypes;
import ru.yandex.webmaster3.api.sitemap.data.ApiSitemapId;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.TypeDescriptionResolver;
import ru.yandex.webmaster3.core.http.request.QueryIdDeserializer;
import ru.yandex.webmaster3.core.http.request.WebmasterHostIdDeserializer;
import ru.yandex.webmaster3.core.http.response.WebmasterHostIdSerializer;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.util.TimeUtils;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * @author avhaliullin
 */
public class WebmasterApiJacksonModule extends SimpleModule {
    private final Map<Class<?>, List<TypeDescriptionResolver>> serializingTypeResolvers = new HashMap<>();
    private final Map<Class<?>, List<TypeDescriptionResolver>> deserializingTypeResolvers = new HashMap<>();

    {
        addDeserializer(URL.class, new ApiURLDeserializer());

        // Тут добавляем resolver'ы для тех типов, которые jackson умеет по дефолту
        addTypeResolver(String.class, simpleResolver(ValueType.STRING));
        addTypeResolver(URL.class, simpleResolver(WebmasterApiTypes.URL));

        addTypeResolver(Byte.class, simpleResolver(ValueType.INT8));
        addTypeResolver(Byte.TYPE, simpleResolver(ValueType.INT8));
        addTypeResolver(Short.class, simpleResolver(ValueType.INT16));
        addTypeResolver(Short.TYPE, simpleResolver(ValueType.INT16));
        addTypeResolver(Integer.class, simpleResolver(ValueType.INT32));
        addTypeResolver(Integer.TYPE, simpleResolver(ValueType.INT32));
        addTypeResolver(Long.class, simpleResolver(ValueType.INT64));
        addTypeResolver(Long.TYPE, simpleResolver(ValueType.INT64));

        addTypeResolver(Float.class, simpleResolver(ValueType.SINGLE));
        addTypeResolver(Float.TYPE, simpleResolver(ValueType.SINGLE));
        addTypeResolver(Double.class, simpleResolver(ValueType.DOUBLE));
        addTypeResolver(Double.TYPE, simpleResolver(ValueType.DOUBLE));

        addTypeResolver(Boolean.class, simpleResolver(ValueType.BOOLEAN));
        addTypeResolver(Boolean.TYPE, simpleResolver(ValueType.BOOLEAN));

        addTypeResolver(Character.class, simpleResolver(WebmasterApiTypes.CHAR));
        addTypeResolver(Character.TYPE, simpleResolver(WebmasterApiTypes.CHAR));
    }

    @Override
    public void setupModule(SetupContext context) {
        super.setupModule(context);
        context.insertAnnotationIntrospector(new WebmasterApiAnnotationIntrospector());
    }

    @Override
    public SimpleModule addSerializer(JsonSerializer<?> ser) {
        if (ser instanceof TypeDescriptionResolver) {
            addSerializingResolver(ser.handledType(), (TypeDescriptionResolver) ser);
        }
        return super.addSerializer(ser);
    }

    @Override
    public <T> SimpleModule addSerializer(Class<? extends T> type, JsonSerializer<T> ser) {
        if (ser instanceof TypeDescriptionResolver) {
            addSerializingResolver(type, (TypeDescriptionResolver) ser);
        }
        return super.addSerializer(type, ser);
    }

    public SimpleModule addSerializer(JsonSerializer<?> ser, TypeDescriptionResolver res) {
        addSerializingResolver(ser.handledType(), res);
        return addSerializer(ser);
    }

    public <T> SimpleModule addSerializer(Class<? extends T> type, JsonSerializer<T> ser, TypeDescriptionResolver res) {
        addSerializingResolver(type, res);
        return addSerializer(type, ser);
    }

    public SimpleModule addSerializingResolver(Class<?> type, TypeDescriptionResolver resolver) {
        serializingTypeResolvers.computeIfAbsent(type, ign -> new ArrayList<>()).add(resolver);
        return this;
    }

    @Override
    public <T> SimpleModule addDeserializer(Class<T> type, JsonDeserializer<? extends T> deser) {
        if (deser instanceof TypeDescriptionResolver) {
            addDeserializingResolver(type, (TypeDescriptionResolver) deser);
        }
        return super.addDeserializer(type, deser);
    }

    public <T> SimpleModule addDeserializer(Class<T> type, JsonDeserializer<? extends T> deser, TypeDescriptionResolver res) {
        addDeserializingResolver(type, res);
        return addDeserializer(type, deser);
    }

    public SimpleModule addDeserializingResolver(Class<?> type, TypeDescriptionResolver resolver) {
        deserializingTypeResolvers.computeIfAbsent(type, ign -> new ArrayList<>()).add(resolver);
        return this;
    }

    public SimpleModule addTypeResolver(Class<?> type, TypeDescriptionResolver resolver) {
        addSerializingResolver(type, resolver);
        addDeserializingResolver(type, resolver);
        return this;
    }

    public <T> SimpleModule addType(Class<T> type, JsonSerializer<? super T> ser, JsonDeserializer<? extends T> deser, TypeDescriptionResolver res) {
        addDeserializer(type, deser, res);
        addSerializer(type, ser, res);
        return this;
    }

    public TypeDescriptionResolver getSerializingTypeResolver() {
        return TypeDescriptionResolver.fromMap(serializingTypeResolvers);
    }

    public TypeDescriptionResolver getDeserializingTypeResolver() {
        return TypeDescriptionResolver.fromMap(deserializingTypeResolvers);
    }

    public static WebmasterApiJacksonModule createDefaultModule() {
        WebmasterApiJacksonModule module = new WebmasterApiJacksonModule();

        ToStringSerializer toStringSer = ToStringSerializer.instance;

        module.addType(UUID.class,
                new UUIDSerializer(),
                new UUIDDeserializer(),
                simpleResolver(WebmasterApiTypes.UUID)
        );

        /* Joda */
        module.addType(
                DateTime.class,
                new DateTimeSerializer(
                        new JacksonJodaDateFormat(FormatConfig.DEFAULT_DATETIME_PRINTER, TimeUtils.ZONE_MSK).withUseTimestamp(false)
                ),
                DateTimeDeserializer.forType(DateTime.class),
                simpleResolver(WebmasterApiTypes.DATE_TIME)
        );

        module.addType(
                LocalDate.class,
                new LocalDateSerializer(FormatConfig.DEFAULT_LOCAL_DATEONLY_FORMAT.withUseTimestamp(false)),
                new LocalDateDeserializer(),
                simpleResolver(WebmasterApiTypes.DATE)
        );

        module.addType(
                Instant.class,
                new InstantSerializer(),
                new InstantDeserializer(),
                simpleResolver(WebmasterApiTypes.TIMESTAMP_MS)
        );

        /* Webmaster */
        module.addType(
                WebmasterHostId.class,
                new WebmasterHostIdSerializer(false),
                new WebmasterHostIdDeserializer(),
                simpleResolver(WebmasterApiTypes.HOST_ID)
        );

        module.addSerializer(ASCIIHostUrl.class, toStringSer, simpleResolver(WebmasterApiTypes.HOST_ASCII_URL));
        module.addSerializer(UnicodeHostUrl.class, toStringSer, simpleResolver(WebmasterApiTypes.HOST_UNICODE_URL));

        module.addSerializer(ApiSitemapId.class, new ApiSitemapIdSerializer(), simpleResolver(WebmasterApiTypes.SITEMAP_ID));

        module.addType(QueryId.class, toStringSer, new QueryIdDeserializer(), simpleResolver(WebmasterApiTypes.QUERY_ID));

        return module;
    }

    private static TypeDescriptionResolver simpleResolver(ValueType type) {
        return ign -> type;
    }
}
