package ru.yandex.webmaster3.api.http.rest.routing;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.dataformat.xml.JacksonXmlModule;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.eclipse.jetty.util.MultiPartInputStreamParser;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.http.MediaType;
import ru.yandex.autodoc.common.doc.view.Markup;
import ru.yandex.autodoc.common.doc.view.renderers.HtmlRenderer;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.ApiRequestAuthorizer;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.common.request.parameters.MaxValueFilter;
import ru.yandex.webmaster3.api.http.common.request.parameters.MinValueFilter;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.http.rest.autodoc.ApiDocumentationBuilder;
import ru.yandex.webmaster3.api.http.rest.autodoc.ObjectModelResolver;
import ru.yandex.webmaster3.api.http.rest.jackson.WebmasterApiJacksonModule;
import ru.yandex.webmaster3.api.http.rest.jackson.ser.RequestAwareSerializer;
import ru.yandex.webmaster3.api.http.rest.jackson.xml.CustomXmlModule;
import ru.yandex.webmaster3.api.http.rest.request.ApiRequest;
import ru.yandex.webmaster3.api.http.rest.request.ApiRequestConverter;
import ru.yandex.webmaster3.api.http.rest.request.ApiRequestFilter;
import ru.yandex.webmaster3.api.http.rest.request.ApiRequestParameterFilter;
import ru.yandex.webmaster3.api.http.rest.request.MatchWithAnnotation;
import ru.yandex.webmaster3.api.http.rest.request.RawStringRequestContent;
import ru.yandex.webmaster3.api.http.rest.request.ResourceLocator;
import ru.yandex.webmaster3.api.http.rest.request.meta.AbstractValidatingHeadersAction;
import ru.yandex.webmaster3.api.http.rest.request.meta.ApiRequestWithEntity;
import ru.yandex.webmaster3.api.http.rest.request.meta.ApiRequestWithRawContent;
import ru.yandex.webmaster3.api.http.rest.response.ApiResponse;
import ru.yandex.webmaster3.api.http.rest.response.HttpStatus;
import ru.yandex.webmaster3.api.http.rest.response.errors.ApiErrorResponse;
import ru.yandex.webmaster3.api.http.rest.response.errors.CommonApiErrors;
import ru.yandex.webmaster3.api.http.rest.response.errors.InternalErrors;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithAllow;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithLocation;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithoutEntity;
import ru.yandex.webmaster3.api.http.rest.types.WebmasterApiTypes;
import ru.yandex.webmaster3.api.http.util.ApiReflectionUtil;
import ru.yandex.webmaster3.core.http.RequestTrace;
import ru.yandex.webmaster3.core.http.RequestTracer;
import ru.yandex.webmaster3.core.http.autodoc.FullTypeInfo;
import ru.yandex.webmaster3.core.http.internal.ParameterInfo;
import ru.yandex.webmaster3.core.http.request.RequestId;
import ru.yandex.webmaster3.core.http.util.RequestFilterHelper;
import ru.yandex.webmaster3.core.metrics.MonitoringCategoryUtil;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimer;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;
import ru.yandex.webmaster3.core.util.ReflectionUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.core.util.functional.ThrowingFunction;

import javax.servlet.MultipartConfigElement;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.Part;
import javax.xml.stream.XMLInputFactory;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.annotation.Annotation;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;

/**
 * @author avhaliullin
 */
public abstract class AbstractApiRouter extends AbstractHandler {
    private static final Logger log = LoggerFactory.getLogger(AbstractApiRouter.class);

    public static final String BALANCER_REQUEST_ID_ATTRIBUTE_NAME = "balancer_request_id";
    public static final String API_ABSOLUTE_BASE_URL_ATTRIBUTE_NAME = "api_base_url";
    public static final String REQUESTED_VERSION_ATTRIBUTE_NAME = "requested_api_version";
    public static final String CURRENT_ACTION_ATTRIBUTE_NAME = "current_action";

    private static final String SOLOMON_LABEL_ACTION = "action";
    private static final String SOLOMON_LABEL_CATEGORY = "category";
    private static final String SOLOMON_LABEL_HTTP_CODE = "http_code";
    private static final String SOLOMON_LABEL_RESULT = "result";
    private static final String SOLOMON_VERSION_NAME = "version";

    private static final String LOCATION_HEADER = "Location";
    private static final String ALLOW_HEADER = "Allow";

    private static final String CONTENT_ENCODING_IDENTITY = "identity";
    private static final String CONTENT_ENCODING_GZIP = "gzip";

    private static final PropertyNamingStrategy.SnakeCaseStrategy PROPERTY_NAMING_STRATEGY =
            new PropertyNamingStrategy.SnakeCaseStrategy();

    private final Map<MediaType, ObjectMapper> SUPPORTED_CONTENT_TYPES = new LinkedHashMap<>();
    private final Map<String, StreamDecoder> SUPPORTED_CONTENT_ENCODINGS = new HashMap<>();
    private final MediaType JSON_CONTENT_TYPE = utf8(MediaType.APPLICATION_JSON);
    private final MediaType XML_CONTENT_TYPE = utf8(MediaType.APPLICATION_XML);

    {
        SUPPORTED_CONTENT_ENCODINGS.put(CONTENT_ENCODING_IDENTITY, is -> is);
        SUPPORTED_CONTENT_ENCODINGS.put(CONTENT_ENCODING_GZIP, is -> new GZIPInputStream(is));
    }

    private final ThreadLocal<Request> REQUEST_THREAD_LOCAL;
    private final WebmasterApiJacksonModule JACKSON_MODULE;
    private final ObjectMapper JSON_OM;
    private final ObjectMapper XML_OM;

    {
        REQUEST_THREAD_LOCAL = new ThreadLocal<>();
        JACKSON_MODULE = WebmasterApiJacksonModule.createDefaultModule();
        JACKSON_MODULE.addSerializer(ResourceLocator.class, new ResourceLocatorSerializer(), ign -> WebmasterApiTypes.URL);
        JSON_OM = setupObjectMapper(createJsonMapper(), JACKSON_MODULE);
        XML_OM = setupObjectMapper(createXmlMapper(), JACKSON_MODULE);

        SUPPORTED_CONTENT_TYPES.put(JSON_CONTENT_TYPE, JSON_OM);
        SUPPORTED_CONTENT_TYPES.put(XML_CONTENT_TYPE, XML_OM);
    }

    private final Map<Class<?>, Resource<?>> resourcesMap = new LinkedHashMap<>();
    private final ApiRequestConverter requestConverter = new ApiRequestConverter(PROPERTY_NAMING_STRATEGY::translate);
    private List<ApiRequestFilter> requestFilters;
    private List<ApiRequestParameterFilter> requestParameterFilters = new ArrayList<>();
    private IdentityHashMap<AbstractApiAction, List<ApiRequestFilter>> action2RequestFilters;
    private IdentityHashMap<AbstractApiAction, Map<String, List<ApiRequestParameterFilter>>> action2Param2Filter;
    private IdentityHashMap<AbstractApiAction, ActionMetric> action2Metric;
    private ActionMetric unknownActionMetric;

    private HierarchicStorage<Resource<?>> routes;
    private ApiRequestAuthorizer<?> apiRequestAuthorizer;
    private ApiDocumentationBuilder apiDocumentationBuilder;
    private HtmlRenderer htmlMarkupRenderer;

    private SolomonMetricRegistry solomonMetricRegistry;
    private SolomonTimerConfiguration solomonTimerConfiguration;

    private String versionName;

    public void init() {
        RequestFilterHelper<ApiRequestFilter> filterHelper = new RequestFilterHelper<>(
                Collections.singleton(ApiRequest.class),
                requestFilters,
                ApiReflectionUtil::getRequestFilterRequestType
        );
        filterHelper.verifyFilterDepsTree();

        setupParamFilters();
        setupRoutes();
        HierarchicStorage.Builder<Resource<?>> routesBuilder = new HierarchicStorage.Builder<>();
        for (Resource<?> resource : resourcesMap.values()) {
            HierarchicStorage.Builder<Resource<?>> cur = routesBuilder;
            for (PathPart<?> pathPart : resource.getPath()) {
                if (pathPart instanceof PathPart.Const<?>) {
                    cur = cur.addStatic(((PathPart.Const) pathPart).getSegment());
                } else {
                    cur = cur.addVariable();
                }
            }
            cur.setValue(resource);
        }
        routes = routesBuilder.build();

        action2Param2Filter = new IdentityHashMap<>();
        action2RequestFilters = new IdentityHashMap<>();

        List<AbstractApiAction> allActions = new ArrayList<>();
        for (Resource<?> resource : resourcesMap.values()) {
            for (AbstractApiAction action : resource.getMethod2Action().values()) {
                allActions.add(action);
                // Request filters
                Class requestClass = ApiReflectionUtil.getActionRequestType(action).getClazz();
                action2RequestFilters.put(action, filterHelper.getFiltersForRequest(requestClass));

                // Param filters
                Map<String, List<ApiRequestParameterFilter>> paramFiltersMap = new HashMap<>();
                List<ParameterInfo> parameters = requestConverter.getAllRequestParameters(requestClass);
                for (ApiRequestParameterFilter parameterFilter : requestParameterFilters) {
                    Class<?> filterClass = parameterFilter.getClass();
                    FullTypeInfo filterParameterType = ApiReflectionUtil.getRequestParameterFilterType(filterClass);
                    for (ParameterInfo parameterInfo : parameters) {
                        Class<?> actualParameterClass = parameterInfo.type.getClazz();
                        if (actualParameterClass.isPrimitive()) {
                            actualParameterClass = ReflectionUtils.primitiveToBoxed(actualParameterClass);
                        }
                        if (filterParameterType.isAssignableFrom(actualParameterClass)) {
                            boolean matches;
                            if (filterClass.isAnnotationPresent(MatchWithAnnotation.class)) {
                                MatchWithAnnotation matchWithAnnotation = filterClass.getAnnotation(MatchWithAnnotation.class);
                                Class<? extends Annotation> requiredAnnotationClass = matchWithAnnotation.annotationClass();
                                matches = parameterInfo.annotations.containsKey(requiredAnnotationClass);
                            } else {
                                matches = true;
                            }
                            if (matches) {
                                paramFiltersMap
                                        .computeIfAbsent(parameterInfo.name, ign -> new ArrayList<>())
                                        .add(parameterFilter);
                            }
                        }
                    }
                }
                if (paramFiltersMap.isEmpty()) {
                    paramFiltersMap = Collections.emptyMap();
                }
                action2Param2Filter.put(action, paramFiltersMap);
            }
        }

        { //Metric
            unknownActionMetric = new ActionMetric("<unknown>", null);
            action2Metric = new IdentityHashMap<>();
            for (var action : allActions) {
                String category = MonitoringCategoryUtil.getCategory(action.getClass()).orElse(null);
                action2Metric.put(action, new ActionMetric(action.getClass().getName(), category));
            }
        }

        List<Class<? extends ApiErrorResponse<?>>> commonResponses = new ArrayList<>();
        commonResponses.add((Class<? extends ApiErrorResponse<?>>)
                ApiReflectionUtil.getAuthorizerResponseClass(apiRequestAuthorizer).getClazz());
        commonResponses.add(CommonApiErrors.class);
        ObjectModelResolver objectModelResolver = new ObjectModelResolver(JACKSON_MODULE, JSON_OM);

        apiDocumentationBuilder = new ApiDocumentationBuilder(
                objectModelResolver,
                action2RequestFilters,
                action2Param2Filter,
                commonResponses,
                requestConverter
        );
        htmlMarkupRenderer = HtmlRenderer.INSTANCE;
    }

    public String getVersionName() {
        return versionName;
    }

    protected void setupParamFilters() {
        addParamFilter(new MaxValueFilter());
        addParamFilter(new MinValueFilter());
    }

    protected abstract void setupRoutes();

    protected void addParamFilter(ApiRequestParameterFilter<?> filter) {
        requestParameterFilters.add(filter);
    }

    protected <L> ResourceLocatorBuilder.PathBuilder0<L> locator(Class<L> locatorClass) {
        return new ResourceLocatorBuilder.PathBuilder0<L>(locatorClass, Collections.emptyList(), Collections.emptyList());
    }

    protected <L> void addResource(ResourceBuilder<L> resource) {
        Class<L> locatorClass = resource.locatorClass;
        if (resourcesMap.containsKey(locatorClass)) {
            throw new IllegalStateException("Already have resource for locator " + locatorClass);
        }
        resourcesMap.put(locatorClass, resourceFromBuilder(resource));
    }

    @Override
    public void handle(String target, Request request, HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException, ServletException {
        try {
            REQUEST_THREAD_LOCAL.set(request);
            RequestTracer.startTrace(true);
            ApiResponse error;
            try {
                error = handleResponseMediaType(request, httpServletResponse);
            } catch (Exception e) {
                log.error("Fatal error - exception during response type handling", e);
                httpServletResponse.setStatus(500);
                request.setHandled(true);
                return;
            }
            try {
                if (error != null) {
                    respond(request, error, httpServletResponse);
                    return;
                }
                QueryString queryString = parseQueryString(target, request);
                Optional<Resource<?>> resourceOpt = routes.getValue(queryString.getPathSegments());
                if (!resourceOpt.isPresent()) {
                    respond(request, new CommonApiErrors.ResourceNotFoundError("Resource not found"), httpServletResponse);
                    return;
                }
                Resource<?> resource = resourceOpt.get();
                handleWithResource(queryString, resource, request, httpServletResponse);
            } catch (Exception e) {
                log.error("Request failed", e);
                respond(request, new InternalErrors.InternalError(), httpServletResponse);
            }
        } finally {
            REQUEST_THREAD_LOCAL.remove();
        }
    }

    public void handleDocumentation(Request request, HttpServletResponse response) throws IOException, ServletException {
        request.setHandled(true);
        response.setContentType("text/html;charset=utf-8");
        Markup documentationMarkup = apiDocumentationBuilder.buildDocumentationMarkup(resourcesMap.values());
        String documentationHtml = htmlMarkupRenderer.render(documentationMarkup);
        response.getWriter().write(documentationHtml);
    }

    private <T> void handleWithResource(QueryString queryString, Resource<T> resource, Request request, HttpServletResponse response) throws IOException {
        T locator;
        try {
            locator = resource.getLocatorMapping().leftToRight(queryString);
        } catch (QueryStringParseException e) {
            respondWithParseQueryError(request, e, response);
            return;
        }

        String method = request.getMethod();
        AbstractApiAction apiAction = resource.getMethod2Action().get(method);
        if (apiAction == null) {
            respond(request, new CommonApiErrors.MethodNotAllowedError(resource.getMethod2Action().keySet()), response);
            return;
        }
        request.setAttribute(CURRENT_ACTION_ATTRIBUTE_NAME, apiAction);

        Class<? extends AbstractApiAction> actionClass = apiAction.getClass();
        FullTypeInfo requestInfoType = ApiReflectionUtil.getActionRequestType(apiAction);

        ApiRequest<T> requestObject;
        try {
            requestObject = (ApiRequest<T>) requestInfoType.getClazz().newInstance();
        } catch (Exception e) {
            throw new RuntimeException("Failed to instantiate request class " + requestInfoType.getClazz() + " for action " + actionClass, e);
        }
        Permission permission = apiAction.getPermission();
        // fallback from annotation
        if (permission == null) {
            ActionPermission annotation = apiAction.getClass().getAnnotation(ActionPermission.class);
            if (annotation != null) {
                permission = annotation.value();
            }
        }
        Preconditions.checkState(permission != null, "Action does not have permission");
        ApiResponse errorResponse = apiRequestAuthorizer.authorize(permission, request, requestObject);
        if (errorResponse != null) {
            respond(request, errorResponse, response);
            return;
        }
        requestObject.setBalancerRequestId(getRequestId(request));
        requestObject.setLocator(locator);

        errorResponse = requestConverter.fillRequestParams(
                action2Param2Filter.get(apiAction),
                requestInfoType,
                requestObject,
                queryString
        );
        if (errorResponse != null) {
            respond(request, errorResponse, response);
            return;
        }

        if (requestObject instanceof ApiRequestWithEntity) {
            FullTypeInfo entityType = requestInfoType.ancestorFullType(ApiRequestWithEntity.class).getGenericsList().get(0);
            if (apiAction instanceof AbstractValidatingHeadersAction) {
                errorResponse = ((AbstractValidatingHeadersAction) apiAction).validateRequestHeaders(request, requestObject);
                if (errorResponse != null) {
                    respond(request, errorResponse, response);
                    return;
                }
            }
            errorResponse = fillEntity(entityType.getClazz(), request, (ApiRequestWithEntity) requestObject);
            if (errorResponse != null) {
                respond(request, errorResponse, response);
                return;
            }
        }

        errorResponse = applyRequestFilters(apiAction, requestObject);
        if (errorResponse != null) {
            respond(request, errorResponse, response);
            return;
        }

        respond(request, apiAction.process(requestObject), response);
    }

    private <T> ApiResponse applyRequestFilters(AbstractApiAction<ApiRequest<T>, ?> action, ApiRequest<T> requestObject) {
        ApiResponse response = null;
        List<ApiRequestFilter> filters = action2RequestFilters.get(action);
        for (ApiRequestFilter filter : filters) {
            response = filter.applyFilter(requestObject);
            if (response != null) {
                return response;
            }
        }
        return response;
    }

    private MediaType parseContentType(String ct) {
        MediaType _parsedCt = null;
        if (ct != null) {
            try {
                _parsedCt = MediaType.parseMediaType(ct);
            } catch (Exception e) {
                //ignored
            }
        }
        return _parsedCt;
    }

    private ApiResponse fillEntity(Class entityType, Request httpRequest, ApiRequestWithEntity apiRequest) {
        MediaType parsedCt = parseContentType(httpRequest.getContentType());//final
        StreamDecoder decoder = getStreamDecoder(httpRequest);
        if (decoder == null) {
            return new CommonApiErrors.UnsupportedContentEncodingError(SUPPORTED_CONTENT_ENCODINGS.keySet());
        }
        if (apiRequest instanceof ApiRequestWithRawContent) {
            ApiRequestWithRawContent customContentRequest = (ApiRequestWithRawContent) apiRequest;
            Set<MediaType> allowedTypes = customContentRequest.allowedMediaTypes();
            boolean matches = allowedTypes.stream().anyMatch(allowedCt -> allowedCt.includes(parsedCt));
            if (!matches) {
                return new CommonApiErrors.UnsupportedMediaTypeError(allowedTypes.stream().map(MediaType::toString).collect(Collectors.toSet()));
            }
            if (MediaType.MULTIPART_FORM_DATA.includes(parsedCt)) {
                try {
                    MultiPartInputStreamParser parser = new MultiPartInputStreamParser(
                            httpRequest.getInputStream(),
                            httpRequest.getContentType(),
                            new MultipartConfigElement(""),
                            new File("/")
                    );
                    Collection<Part> parts = parser.getParts();
                    Set<MediaType> allowedPartTypes = customContentRequest.allowedPartMediaTypes();
                    boolean foundMatch = false;
                    for (Part part : parts) {
                        MediaType parsedPartCt = parseContentType(part.getContentType());
                        if (parsedPartCt == null) {
                            continue;
                        }
                        foundMatch = allowedPartTypes.stream().anyMatch(allowedPartCt -> allowedPartCt.includes(parsedPartCt));
                        if (foundMatch) {
                            customContentRequest.setEntity(new RawStringRequestContent(ByteStreams.toByteArray(part.getInputStream()), parsedCt));
                            break;
                        }
                    }
                    if (!foundMatch) {
                        return new CommonApiErrors.UnsupportedMediaTypeError(
                                allowedPartTypes.stream().map(MediaType::toString).collect(Collectors.toSet()),
                                "Content-Type of multipart's part not supported"
                        );
                    }
                } catch (IOException | ServletException e) {
                    log.warn("Multipart entity processing error", e);
                    return new CommonApiErrors.EntityValidationError("Cannot parse request entity");
                }
            } else {
                try (InputStream in = decoder.apply(httpRequest.getInputStream())) {
                    byte[] data = ByteStreams.toByteArray(in);
                    customContentRequest.setEntity(new RawStringRequestContent(data, parsedCt));
                } catch (IOException e) {
                    log.warn("Entity processing error", e);
                    return new CommonApiErrors.EntityValidationError("Cannot parse request entity");
                }
            }
        } else {
            ObjectMapper objectMapper = parsedCt == null ? null : getInputMapper(parsedCt);
            if (objectMapper == null) {
                return new CommonApiErrors.UnsupportedMediaTypeError(SUPPORTED_CONTENT_TYPES.keySet().stream().map(MediaType::toString).collect(Collectors.toSet()));
            }
            Charset charset = parsedCt.getCharset() == null ? StandardCharsets.UTF_8 : parsedCt.getCharset();
            Object entity;
            try (Reader reader = new InputStreamReader(decoder.apply(httpRequest.getInputStream()), charset)) {
                entity = objectMapper.readValue(reader, entityType);
            } catch (IOException e) {
                log.warn("Entity processing error", e);
                //TODO: WMC-3014 Кажется, десериализацию придется изобретать самостоятельно
                // Выключил отправку jackson'овской ошибки, поскольку это небезопасно
                return new CommonApiErrors.EntityValidationError("Cannot parse request entity");
            }
            apiRequest.setEntity(entity);
        }
        return null;
    }

    private StreamDecoder getStreamDecoder(Request httpRequest) {
        String encoding = httpRequest.getHeader("Content-Encoding");
        encoding = StringUtils.isEmpty(encoding) ? CONTENT_ENCODING_IDENTITY : encoding;
        return SUPPORTED_CONTENT_ENCODINGS.get(encoding);
    }

    private void respondWithParseQueryError(Request request, QueryStringParseException e, HttpServletResponse httpResponse) throws IOException {
        if (e instanceof QueryStringParseException.MissingRequiredParamException) {
            respond(request, new CommonApiErrors.FieldValidationError("This field is required", e.getParamName(), e.getPassedValue()), httpResponse);
        } else {
            String message = Optional.ofNullable(e.getCause()).map(Throwable::getMessage).orElse("Validation error");
            respond(request, new CommonApiErrors.FieldValidationError(message, e.getParamName(), e.getPassedValue()), httpResponse);
        }
    }

    private void respond(Request request, ApiResponse apiResponse, HttpServletResponse httpResponse) throws IOException {
        handleInputOnClose(request);
        MediaType resultType = MediaType.parseMediaType(httpResponse.getContentType());
        httpResponse.setStatus(apiResponse.getStatus().getCode());
        handleResponseMeta(request, apiResponse, httpResponse);
        if (httpResponse.getContentType() != null) {
            SUPPORTED_CONTENT_TYPES.get(resultType).writeValue(httpResponse.getWriter(), apiResponse);
        }
        // Metric
        AbstractApiAction apiAction = getRequestAttribute(request, AbstractApiAction.class, CURRENT_ACTION_ATTRIBUTE_NAME);
        RequestTrace requestTrace = RequestTracer.stopTrace();
        log.info("Request trace: {} cr {} crt {} cw {} cwt {} cb {} cbt {}",
                apiAction == null ? null : apiAction.getClass().getSimpleName(),
                requestTrace.getCassandraReadCount(), requestTrace.getTotalCassandraReadTimeNano(TimeUnit.MILLISECONDS),
                requestTrace.getCassandraWriteCount(), requestTrace.getTotalCassandraWriteTimeNano(TimeUnit.MILLISECONDS),
                requestTrace.getCassandraBatchCount(), requestTrace.getTotalCassandraBatchTimeNano(TimeUnit.MILLISECONDS)
        );
        var actionMetric = action2Metric.getOrDefault(apiAction, unknownActionMetric);
        var duration = Duration.millis(requestTrace.getRequestDuration(TimeUnit.MILLISECONDS));
        actionMetric.update(duration, apiResponse.getStatus().getCode());
    }

    private void handleInputOnClose(Request request) {
        if (!request.getHttpInput().isEOF()) {
            if (!"100-continue".equals(request.getHeader("Expect"))) {
                request.getHttpInput().consumeAll();
            }
        }
    }

    private void handleResponseMeta(Request request, ApiResponse apiResponse, HttpServletResponse httpResponse) {
        if (apiResponse instanceof ResponseWithLocation) {
            String location = locatorToRelUrl(((ResponseWithLocation) apiResponse).getLocation());
            if (location != null) {
                httpResponse.addHeader(LOCATION_HEADER, getApiBaseUrl(request) + location);
            }
        }
        if (apiResponse instanceof ResponseWithAllow) {
            String value = String.join(", ", ((ResponseWithAllow) apiResponse).getAllowedMethods());
            httpResponse.addHeader(ALLOW_HEADER, value);
        }
        if (apiResponse instanceof ResponseWithoutEntity) {
            httpResponse.setContentType(null);
        }
    }

    private String locatorToRelUrl(Object locator) {
        Resource resource = resourcesMap.get(locator.getClass());
        if (resource == null) {
            log.error("Resource not found for locator " + locator.getClass());
            return null;
        }
        QueryString queryString = (QueryString) resource.getLocatorMapping().rightToLeft(locator);
        return queryString.toString();
    }

    private QueryString parseQueryString(String target, Request request) {
        if (!target.endsWith("/")) {
            target = target + "/";
        }
        List<String> pathSegments = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i < target.length(); i++) {
            char ch = target.charAt(i);
            if (ch == '/') {
                try {
                    pathSegments.add(URLEncodeUtil.prettifyUrl(sb.toString()));
                } catch (Exception e) {
                    log.warn("Failed to urldecode path part " + sb.toString() + " in " + target);
                    pathSegments.add(sb.toString());
                }
                sb.setLength(0);
            } else {
                sb.append(ch);
            }
        }

        Map<String, List<String>> query = new HashMap<>();
        if (request.getParameterMap() != null) {
            for (Map.Entry<String, String[]> entry : request.getParameterMap().entrySet()) {
                if (entry.getValue() != null && entry.getValue().length > 0) {
                    query.put(entry.getKey(), new ArrayList<>(Arrays.asList(entry.getValue())));
                }
            }
        }
        return new QueryString(pathSegments, query);
    }

    private <T> Resource<T> resourceFromBuilder(ResourceBuilder<T> builder) {
        Map<String, AbstractApiAction> method2Action = new HashMap<>();
        if (builder.onGet != null) {
            method2Action.put("GET", builder.onGet);
        }
        if (builder.onDelete != null) {
            method2Action.put("DELETE", builder.onDelete);
        }
        if (builder.onPost != null) {
            method2Action.put("POST", builder.onPost);
        }
        return new Resource<>(builder.locatorClass, method2Action, builder.path, builder.query, builder.locatorMapping);
    }

    private ApiResponse handleResponseMediaType(Request request, HttpServletResponse response) {
        String acceptHeader = request.getHeader("Accept");
        MediaType contentType = getResponseType(acceptHeader);
        ApiResponse result = null;
        if (contentType == null) {
            contentType = JSON_CONTENT_TYPE;
            result = new CommonApiErrors.NotAcceptableError(SUPPORTED_CONTENT_TYPES.keySet().stream().map(Object::toString).collect(Collectors.toSet()));
        }
        response.setContentType(contentType.toString());
        return result;
    }

    /**
     * Все так сложно на всякий случай - если мы, например, поддержим обращение по минорной версии,
     * то в Location нужно будет возвращать ту же версию, что в запросе, а не точную версию текущего api
     */
    private static String getApiBaseUrl(Request request) {
        return (String) request.getAttribute(API_ABSOLUTE_BASE_URL_ATTRIBUTE_NAME);
    }

    private static RequestId getRequestId(Request request) {
        return (RequestId) request.getAttribute(BALANCER_REQUEST_ID_ATTRIBUTE_NAME);
    }

    private static <T> T getRequestAttribute(Request request, Class<T> clazz, String attributeName) {
        Object value = request.getAttribute(attributeName);
        if (value == null) {
            return null;
        }
        if (!clazz.isAssignableFrom(value.getClass())) {
            log.error("Wrong attribute type: expected {} but found {}", clazz, value.getClass());
            return null;
        }
        return (T) value;
    }

    private static MediaType utf8(MediaType mediaType) {
        Map<String, String> props = new HashMap<>(mediaType.getParameters());
        props.put("charset", "UTF-8");
        return new MediaType(mediaType.getType(), mediaType.getSubtype(), props);
    }

    private MediaType getResponseType(String acceptHeader) {
        if (acceptHeader != null) {
            List<MediaType> acceptableTypes;
            try {
                acceptableTypes = MediaType.parseMediaTypes(acceptHeader);
            } catch (Exception e) {
                return null;
            }
            MediaType.sortByQualityValue(acceptableTypes);
            for (MediaType acceptableRange : acceptableTypes) {
                for (MediaType supportedType : SUPPORTED_CONTENT_TYPES.keySet()) {
                    if (acceptableRange.includes(supportedType)) {
                        return supportedType;
                    }
                }
            }
        }
        return JSON_CONTENT_TYPE;
    }

    private ObjectMapper getInputMapper(MediaType inputType) {
        for (Map.Entry<MediaType, ObjectMapper> entry : SUPPORTED_CONTENT_TYPES.entrySet()) {
            MediaType supportedRange = entry.getKey();
            if (supportedRange.includes(inputType)) {
                return entry.getValue();
            }
        }
        return null;
    }

    private static ObjectMapper createJsonMapper() {
        return new ObjectMapper();
    }

    private static ObjectMapper createXmlMapper() {
        JacksonXmlModule xmlModule = new CustomXmlModule();
        xmlModule.setDefaultUseWrapper(false);
        XmlMapper om = new XmlMapper(xmlModule);

        XMLInputFactory inputFactory = om.getFactory().getXMLInputFactory();
        inputFactory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
        inputFactory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false); // И так форсится jackson'ом, но лучше перебдеть.
        inputFactory.setXMLResolver((publicID, systemID, baseURI, namespace) -> ""); // Чтобы точно ничего никуда не ресолвилось

        om.enable(ToXmlGenerator.Feature.WRITE_XML_1_1);
        om.enable(ToXmlGenerator.Feature.WRITE_XML_DECLARATION);
        om.setConfig(om.getSerializationConfig().withRootName("Data"));
        return om;
    }

    private static ObjectMapper setupObjectMapper(ObjectMapper om, WebmasterApiJacksonModule webmasterApiModule) {
        return om.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
                .enable(SerializationFeature.WRITE_NULL_MAP_VALUES)
                .enable(DeserializationFeature.FAIL_ON_NULL_CREATOR_PROPERTIES)
                .setSerializationInclusion(JsonInclude.Include.ALWAYS)
                .registerModule(webmasterApiModule)
                .registerModule(new Jdk8Module())
                .registerModule(new ParameterNamesModule());
    }

    private class ActionMetric {
        private final ConcurrentHashMap<Integer, SolomonTimer> metrics = new ConcurrentHashMap<>();
        private final SolomonKey baseKey;

        public ActionMetric(String actionName, String category) {
            this.baseKey = SolomonKey.create(SOLOMON_LABEL_ACTION, actionName)
                    .withLabel(SOLOMON_VERSION_NAME, versionName)
                    .withLabel(SOLOMON_LABEL_CATEGORY, category == null ? "<unknown>" : category);
            for (HttpStatus httpStatus : ApiReflectionUtil.STATUSES_MAP.values()) {
                int code = httpStatus.getCode();
                metrics.computeIfAbsent(code, ign -> {
                            var solomonKey = baseKey.withLabel(SOLOMON_LABEL_HTTP_CODE, String.valueOf(code))
                                    .withLabel(SOLOMON_LABEL_RESULT, resultFromHttpCode(code));
                            return solomonMetricRegistry.createTimer(solomonTimerConfiguration, solomonKey);
                        }
                );
            }
        }

        private String resultFromHttpCode(int code) {
            if (code < 400) {
                return "success";
            } else if (code < 500) {
                return "user_error";
            } else {
                return "internal_error";
            }
        }

        void update(Duration duration, int code) {
            metrics.computeIfAbsent(code, ign ->
                    {
                        var key = baseKey.withLabel(SOLOMON_LABEL_HTTP_CODE, String.valueOf(code))
                                .withLabel(SOLOMON_LABEL_RESULT, resultFromHttpCode(code));
                        return solomonMetricRegistry.createTimer(solomonTimerConfiguration, key);
                    }
            ).update(duration);
        }
    }

    private class ResourceLocatorSerializer extends RequestAwareSerializer<Object> {
        public ResourceLocatorSerializer() {
            super(Object.class, REQUEST_THREAD_LOCAL);
        }

        @Override
        public void serialize(Object value, JsonGenerator gen, SerializerProvider provider, Request request) throws IOException {
            gen.writeString(getApiBaseUrl(request) + locatorToRelUrl(value));
        }
    }

    private interface StreamDecoder extends ThrowingFunction<InputStream, InputStream, IOException> {}

    @Required
    public void setVersionName(String versionName) {
        this.versionName = versionName;
    }

    @Required
    public void setApiRequestAuthorizer(ApiRequestAuthorizer<?> apiRequestAuthorizer) {
        this.apiRequestAuthorizer = apiRequestAuthorizer;
    }

    @Required
    public void setRequestFilters(List<ApiRequestFilter> requestFilters) {
        this.requestFilters = requestFilters;
    }

    @Required
    public void setSolomonMetricRegistry(SolomonMetricRegistry solomonMetricRegistry) {
        this.solomonMetricRegistry = solomonMetricRegistry;
    }

    @Required
    public void setSolomonTimerConfiguration(SolomonTimerConfiguration solomonTimerConfiguration) {
        this.solomonTimerConfiguration = solomonTimerConfiguration;
    }
}
